<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class CategoryController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Mostrar lista de categorías
     */
    public function index()
    {
        return view('administrador.categories.index');
    }

    /**
     * Obtener categorías para DataTable (AJAX)
     */
    public function getData()
    {
        try {
            $categories = Category::withCount('products')
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $categories->map(function ($category) {
                    return [
                        'id' => $category->id,
                        'name' => $category->name,
                        'description' => $category->description,
                        'products_count' => $category->products_count ?? 0,
                        'created_at' => $category->created_at->format('d/m/Y H:i')
                    ];
                })
            ]);
        } catch (\Exception $e) {
            \Log::error('Error al cargar categorías: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar las categorías: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mostrar formulario de crear categoría
     */
    public function create()
    {
        return view('administrador.categories.create');
    }

    /**
     * Guardar nueva categoría (AJAX)
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255|unique:categories,name',
                'description' => 'nullable|string|max:500'
            ], [
                'name.required' => 'El nombre de la categoría es obligatorio',
                'name.unique' => 'Ya existe una categoría con este nombre',
                'name.max' => 'El nombre no puede tener más de 255 caracteres',
                'description.max' => 'La descripción no puede tener más de 500 caracteres'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error de validación',
                    'errors' => $validator->errors()
                ], 422);
            }

            $category = Category::create([
                'name' => $request->name,
                'description' => $request->description
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Categoría creada exitosamente',
                'data' => $category
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al crear la categoría: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mostrar categoría específica
     */
    public function show($id)
    {
        try {
            $category = Category::with('products')->findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => $category
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Categoría no encontrada'
            ], 404);
        }
    }

    /**
     * Mostrar formulario de editar categoría
     */
    public function edit($id)
    {
        try {
            $category = Category::findOrFail($id);
            return view('administrador.categories.edit', compact('category'));
        } catch (\Exception $e) {
            return redirect()->route('admin.categories.index')
                ->with('error', 'Categoría no encontrada');
        }
    }

    /**
     * Actualizar categoría (AJAX)
     */
    public function update(Request $request, $id)
    {
        try {
            $category = Category::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255|unique:categories,name,' . $id,
                'description' => 'nullable|string|max:500'
            ], [
                'name.required' => 'El nombre de la categoría es obligatorio',
                'name.unique' => 'Ya existe una categoría con este nombre',
                'name.max' => 'El nombre no puede tener más de 255 caracteres',
                'description.max' => 'La descripción no puede tener más de 500 caracteres'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error de validación',
                    'errors' => $validator->errors()
                ], 422);
            }

            $category->update([
                'name' => $request->name,
                'description' => $request->description
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Categoría actualizada exitosamente',
                'data' => $category
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar la categoría: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Eliminar categoría (AJAX)
     */
    public function destroy($id)
    {
        try {
            $category = Category::findOrFail($id);

            // Verificar si tiene productos asociados
            if ($category->products()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se puede eliminar la categoría porque tiene productos asociados'
                ], 400);
            }

            $category->delete();

            return response()->json([
                'success' => true,
                'message' => 'Categoría eliminada exitosamente'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar la categoría: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener categorías para select (AJAX)
     */
    public function getForSelect()
    {
        try {
            $categories = Category::orderBy('name')->get(['id', 'name']);
            
            return response()->json([
                'success' => true,
                'data' => $categories
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar las categorías'
            ], 500);
        }
    }
}
