<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CompanyInfo;
use Illuminate\Http\Request;

class CompanyInfoController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Mostrar la vista de información de la empresa
     */
    public function index()
    {
        return view('administrador.company-info.index');
    }

    /**
     * Obtener la información actual de la empresa
     */
    public function getData()
    {
        try {
            // Obtener el primer registro o crear uno vacío si no existe
            $companyInfo = CompanyInfo::first();
            
            if (!$companyInfo) {
                // Crear registro por defecto si no existe
                $companyInfo = CompanyInfo::create([
                    'type' => 'general',
                    'name' => '',
                    'phone' => '',
                    'email' => '',
                    'address' => '',
                    'hours' => '',
                    'description' => '',
                    'is_active' => true
                ]);
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $companyInfo->id,
                    'type' => $companyInfo->type,
                    'name' => $companyInfo->name,
                    'phone' => $companyInfo->phone,
                    'email' => $companyInfo->email,
                    'address' => $companyInfo->address,
                    'hours' => $companyInfo->hours,
                    'description' => $companyInfo->description,
                    'is_active' => $companyInfo->is_active,
                    'created_at' => $companyInfo->created_at->format('d/m/Y H:i'),
                    'updated_at' => $companyInfo->updated_at->format('d/m/Y H:i'),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar la información: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar la información de la empresa
     */
    public function update(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:20',
                'email' => 'required|email|max:255',
                'address' => 'required|string|max:500',
                'hours' => 'required|string|max:1000',
                'description' => 'nullable|string|max:1000',
                'is_active' => 'boolean'
            ]);

            // Obtener el primer registro
            $companyInfo = CompanyInfo::first();
            
            if (!$companyInfo) {
                // Crear si no existe
                $companyInfo = new CompanyInfo();
                $companyInfo->type = 'general';
            }

            // Actualizar campos
            $companyInfo->name = $request->name;
            $companyInfo->phone = $request->phone;
            $companyInfo->email = $request->email;
            $companyInfo->address = $request->address;
            $companyInfo->hours = $request->hours;
            $companyInfo->description = $request->description ?? '';
            $companyInfo->is_active = $request->has('is_active') ? true : false;
            
            $companyInfo->save();

            return response()->json([
                'success' => true,
                'message' => 'Información de la empresa actualizada exitosamente',
                'data' => [
                    'id' => $companyInfo->id,
                    'type' => $companyInfo->type,
                    'name' => $companyInfo->name,
                    'phone' => $companyInfo->phone,
                    'email' => $companyInfo->email,
                    'address' => $companyInfo->address,
                    'hours' => $companyInfo->hours,
                    'description' => $companyInfo->description,
                    'is_active' => $companyInfo->is_active,
                    'updated_at' => $companyInfo->updated_at->format('d/m/Y H:i'),
                ]
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Datos inválidos',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar la información: ' . $e->getMessage()
            ], 500);
        }
    }
}
