<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use App\Models\Category;
use App\Models\ContactMessage;
use App\Models\User;
use App\Models\BlogPost;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Mostrar dashboard principal
     */
    public function index()
    {
        return view('administrador.index');
    }

    /**
     * Obtener estadísticas generales
     */
    public function getStats(Request $request)
    {
        try {
            $dateFrom = $request->get('date_from', Carbon::now()->startOfMonth()->format('Y-m-d'));
            $dateTo = $request->get('date_to', Carbon::now()->format('Y-m-d'));
            
            // Estadísticas de pedidos (con manejo de errores)
            try {
                $ordersQuery = Order::whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59']);
                $totalOrders = $ordersQuery->count();
                $totalSales = $ordersQuery->sum('total') ?? 0;
                $pendingOrders = Order::where('status', 'pending')->count();
                $confirmedOrders = Order::where('status', 'confirmed')->count();
                $processingOrders = Order::where('status', 'processing')->count();
                $shippedOrders = Order::where('status', 'shipped')->count();
                $deliveredOrders = Order::where('status', 'delivered')->count();
                $cancelledOrders = Order::where('status', 'cancelled')->count();
            } catch (\Exception $e) {
                $totalOrders = 0;
                $totalSales = 0;
                $pendingOrders = 0;
                $confirmedOrders = 0;
                $processingOrders = 0;
                $shippedOrders = 0;
                $deliveredOrders = 0;
                $cancelledOrders = 0;
            }
            
            // Estadísticas de productos (con manejo de errores)
            try {
                $totalProducts = Product::count();
                $activeProducts = Product::count(); // No hay campo status, todos son activos
                $inactiveProducts = 0; // No hay productos inactivos
                $totalCategories = Category::count();
            } catch (\Exception $e) {
                $totalProducts = 0;
                $activeProducts = 0;
                $inactiveProducts = 0;
                $totalCategories = 0;
            }
            
            // Estadísticas de usuarios (con manejo de errores)
            try {
                $totalClients = User::where('rol', '!=', 'administrador')->count();
                $totalAdmins = User::where('rol', 'administrador')->count();
                $newClients = User::where('rol', '!=', 'administrador')
                               ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
                               ->count();
                $totalUsers = User::count();
            } catch (\Exception $e) {
                $totalClients = 0;
                $totalAdmins = 0;
                $newClients = 0;
                $totalUsers = 0;
            }
            
            // Estadísticas de mensajes (con manejo de errores)
            try {
                $totalMessages = ContactMessage::count();
                $newMessages = ContactMessage::where('status', 'nuevo')->count();
                $readMessages = ContactMessage::where('status', 'leido')->count();
                $answeredMessages = ContactMessage::where('status', 'respondido')->count();
            } catch (\Exception $e) {
                $totalMessages = 0;
                $newMessages = 0;
                $readMessages = 0;
                $answeredMessages = 0;
            }
            
            // Estadísticas del blog (con manejo de errores)
            try {
                $totalPosts = BlogPost::count();
                $featuredPosts = BlogPost::where('featured', true)->count();
                $newPosts = BlogPost::whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])->count();
            } catch (\Exception $e) {
                $totalPosts = 0;
                $featuredPosts = 0;
                $newPosts = 0;
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'orders' => [
                        'total' => $totalOrders,
                        'pending' => $pendingOrders,
                        'confirmed' => $confirmedOrders,
                        'processing' => $processingOrders,
                        'shipped' => $shippedOrders,
                        'delivered' => $deliveredOrders,
                        'cancelled' => $cancelledOrders,
                        'total_sales' => number_format($totalSales, 2)
                    ],
                    'products' => [
                        'total' => $totalProducts,
                        'active' => $activeProducts,
                        'inactive' => $inactiveProducts,
                        'categories' => $totalCategories
                    ],
                    'users' => [
                        'total' => $totalUsers,
                        'clients' => $totalClients,
                        'admins' => $totalAdmins,
                        'new' => $newClients
                    ],
                    'messages' => [
                        'total' => $totalMessages,
                        'new' => $newMessages,
                        'read' => $readMessages,
                        'answered' => $answeredMessages
                    ],
                    'blog' => [
                        'total' => $totalPosts,
                        'featured' => $featuredPosts,
                        'new' => $newPosts
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar estadísticas: ' . $e->getMessage(),
                'data' => [
                    'orders' => [
                        'total' => 0,
                        'pending' => 0,
                        'confirmed' => 0,
                        'processing' => 0,
                        'shipped' => 0,
                        'delivered' => 0,
                        'cancelled' => 0,
                        'total_sales' => '0.00'
                    ],
                    'products' => [
                        'total' => 0,
                        'active' => 0,
                        'inactive' => 0,
                        'categories' => 0
                    ],
                    'users' => [
                        'total' => 0,
                        'clients' => 0,
                        'admins' => 0,
                        'new' => 0
                    ],
                    'messages' => [
                        'total' => 0,
                        'new' => 0,
                        'read' => 0,
                        'answered' => 0
                    ],
                    'blog' => [
                        'total' => 0,
                        'featured' => 0,
                        'new' => 0
                    ]
                ]
            ], 200);
        }
    }

    /**
     * Obtener datos para gráfico de ventas por día
     */
    public function getSalesChart(Request $request)
    {
        try {
            $days = $request->get('days', 30);
            $startDate = Carbon::now()->subDays($days);
            
            $salesData = Order::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('COUNT(*) as orders'),
                DB::raw('SUM(total) as total_sales')
            )
            ->where('created_at', '>=', $startDate)
            ->groupBy('date')
            ->orderBy('date')
            ->get();
            
            $labels = [];
            $orders = [];
            $sales = [];
            
            // Llenar datos para todos los días
            for ($i = $days - 1; $i >= 0; $i--) {
                $date = Carbon::now()->subDays($i)->format('Y-m-d');
                $labels[] = Carbon::now()->subDays($i)->format('d/m');
                
                $dayData = $salesData->where('date', $date)->first();
                $orders[] = $dayData ? $dayData->orders : 0;
                $sales[] = $dayData ? (float)$dayData->total_sales : 0;
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'labels' => $labels,
                    'orders' => $orders,
                    'sales' => $sales
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar gráfico de ventas: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener datos para gráfico de estados de pedidos
     */
    public function getOrdersStatusChart(Request $request)
    {
        try {
            $dateFrom = $request->get('date_from', Carbon::now()->startOfMonth()->format('Y-m-d'));
            $dateTo = $request->get('date_to', Carbon::now()->format('Y-m-d'));
            
            $statusData = Order::select('status', DB::raw('COUNT(*) as count'))
                ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
                ->groupBy('status')
                ->get();
            
            $labels = [];
            $data = [];
            $colors = [
                'pending' => '#ed8936',
                'confirmed' => '#4299e1',
                'processing' => '#667eea',
                'shipped' => '#718096',
                'delivered' => '#48bb78',
                'cancelled' => '#f56565'
            ];
            
            $statusLabels = [
                'pending' => 'Pendientes',
                'confirmed' => 'Confirmados',
                'processing' => 'En Preparación',
                'shipped' => 'Enviados',
                'delivered' => 'Entregados',
                'cancelled' => 'Cancelados'
            ];
            
            foreach ($statusData as $status) {
                $labels[] = $statusLabels[$status->status] ?? $status->status;
                $data[] = $status->count;
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'labels' => $labels,
                    'data' => $data,
                    'colors' => array_values($colors)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar gráfico de estados: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener productos más vendidos
     */
    public function getTopProducts(Request $request)
    {
        try {
            $limit = $request->get('limit', 10);
            $dateFrom = $request->get('date_from', Carbon::now()->startOfMonth()->format('Y-m-d'));
            $dateTo = $request->get('date_to', Carbon::now()->format('Y-m-d'));
            
            $topProducts = OrderItem::select(
                'product_name',
                DB::raw('SUM(quantity) as total_quantity'),
                DB::raw('SUM(subtotal) as total_sales')
            )
            ->whereHas('order', function($query) use ($dateFrom, $dateTo) {
                $query->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59']);
            })
            ->groupBy('product_name')
            ->orderBy('total_quantity', 'desc')
            ->limit($limit)
            ->get();
            
            return response()->json([
                'success' => true,
                'data' => $topProducts
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar productos más vendidos: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener ventas por categoría
     */
    public function getCategorySales(Request $request)
    {
        try {
            $dateFrom = $request->get('date_from', Carbon::now()->startOfMonth()->format('Y-m-d'));
            $dateTo = $request->get('date_to', Carbon::now()->format('Y-m-d'));
            
            $categorySales = DB::table('order_items')
                ->join('orders', 'order_items.order_id', '=', 'orders.id')
                ->join('products', 'order_items.product_id', '=', 'products.id')
                ->join('categories', 'products.category_id', '=', 'categories.id')
                ->select(
                    'categories.name as category_name',
                    DB::raw('SUM(order_items.quantity) as total_quantity'),
                    DB::raw('SUM(order_items.subtotal) as total_sales')
                )
                ->whereBetween('orders.created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
                ->groupBy('categories.id', 'categories.name')
                ->orderBy('total_sales', 'desc')
                ->get();
            
            return response()->json([
                'success' => true,
                'data' => $categorySales
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar ventas por categoría: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener pedidos recientes
     */
    public function getRecentOrders(Request $request)
    {
        try {
            $limit = $request->get('limit', 10);
            
            $recentOrders = Order::with('user')
                ->orderBy('created_at', 'desc')
                ->limit($limit)
                ->get()
                ->map(function($order) {
                    return [
                        'id' => $order->id,
                        'order_number' => $order->order_number,
                        'customer_name' => $order->full_name,
                        'customer_email' => $order->user ? $order->user->email : 'Sin email',
                        'total' => number_format($order->total, 2),
                        'status' => $order->status,
                        'status_label' => $order->status_label,
                        'status_class' => $order->status_class,
                        'created_at' => $order->created_at->format('d/m/Y H:i')
                    ];
                });
            
            return response()->json([
                'success' => true,
                'data' => $recentOrders
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar pedidos recientes: ' . $e->getMessage()
            ], 500);
        }
    }
}
