<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ContactMessage;
use Illuminate\Http\Request;

class MessageController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Mostrar la vista principal de mensajes
     */
    public function index()
    {
        return view('administrador.messages.index');
    }

    /**
     * Obtener datos de mensajes para AJAX
     */
    public function getData()
    {
        try {
            $messages = ContactMessage::orderBy('created_at', 'desc')->get();
            
            return response()->json([
                'success' => true,
                'data' => $messages->map(function($message) {
                    return [
                        'id' => $message->id,
                        'name' => $message->name,
                        'email' => $message->email,
                        'phone' => $message->phone,
                        'subject' => $message->subject,
                        'subject_name' => $message->subject_name,
                        'message' => $message->message,
                        'status' => $message->status,
                        'status_class' => $message->status_class,
                        'created_at' => $message->created_at->format('d/m/Y H:i')
                    ];
                })
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar los mensajes'
            ], 500);
        }
    }

    /**
     * Mostrar un mensaje específico
     */
    public function show($id)
    {
        try {
            $message = ContactMessage::findOrFail($id);
            
            // Marcar como leído si está nuevo
            if ($message->status === 'nuevo') {
                $message->markAsRead();
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $message->id,
                    'name' => $message->name,
                    'email' => $message->email,
                    'phone' => $message->phone,
                    'subject' => $message->subject,
                    'subject_name' => $message->subject_name,
                    'message' => $message->message,
                    'status' => $message->status,
                    'created_at' => $message->created_at->format('d/m/Y H:i')
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Mensaje no encontrado'
            ], 404);
        }
    }

    /**
     * Eliminar un mensaje
     */
    public function destroy($id)
    {
        try {
            $message = ContactMessage::findOrFail($id);
            $message->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Mensaje eliminado exitosamente'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el mensaje'
            ], 500);
        }
    }

    /**
     * Marcar mensaje como respondido
     */
    public function markAsAnswered($id)
    {
        try {
            $message = ContactMessage::findOrFail($id);
            $message->markAsAnswered();
            
            return response()->json([
                'success' => true,
                'message' => 'Mensaje marcado como respondido'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el estado'
            ], 500);
        }
    }
}
