<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\User;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Mostrar la vista principal de pedidos
     */
    public function index()
    {
        return view('administrador.orders.index');
    }

    /**
     * Obtener datos de pedidos para AJAX
     */
    public function getData()
    {
        try {
            $orders = Order::with(['user', 'orderItems'])
                ->orderBy('created_at', 'desc')
                ->get();
            
            return response()->json([
                'success' => true,
                'data' => $orders->map(function($order) {
                    return [
                        'id' => $order->id,
                        'order_number' => $order->order_number,
                        'full_name' => $order->full_name,
                        'user_name' => $order->user ? $order->user->name : 'Usuario eliminado',
                        'user_email' => $order->user ? $order->user->email : 'N/A',
                        'phone' => $order->phone,
                        'address' => $order->address,
                        'district' => $order->district,
                        'payment_method' => $order->payment_method,
                        'subtotal' => number_format($order->subtotal, 2),
                        'shipping_cost' => number_format($order->shipping_cost, 2),
                        'total' => number_format($order->total, 2),
                        'status' => $order->status,
                        'status_class' => $this->getStatusClass($order->status),
                        'items_count' => $order->orderItems->count(),
                        'created_at' => $order->created_at->format('d/m/Y H:i')
                    ];
                })
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar los pedidos'
            ], 500);
        }
    }

    /**
     * Mostrar un pedido específico
     */
    public function show($id)
    {
        try {
            $order = Order::with(['user', 'orderItems.product'])->findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'full_name' => $order->full_name,
                    'user' => $order->user ? [
                        'name' => $order->user->name,
                        'email' => $order->user->email
                    ] : null,
                    'phone' => $order->phone,
                    'address' => $order->address,
                    'district' => $order->district,
                    'payment_method' => $order->payment_method,
                    'subtotal' => $order->subtotal,
                    'shipping_cost' => $order->shipping_cost,
                    'total' => $order->total,
                    'status' => $order->status,
                    'created_at' => $order->created_at->format('d/m/Y H:i'),
                    'items' => $order->orderItems->map(function($item) {
                        return [
                            'product_name' => $item->product_name,
                            'product_price' => $item->product_price,
                            'quantity' => $item->quantity,
                            'subtotal' => $item->subtotal
                        ];
                    })
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Pedido no encontrado'
            ], 404);
        }
    }

    /**
     * Actualizar estado del pedido
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            $order = Order::findOrFail($id);
            
            // Validar que el status sea válido (usar los mismos valores de la base de datos)
            $validStatuses = Order::getValidStatuses();
            
            $validator = \Validator::make($request->all(), [
                'status' => 'required|in:' . implode(',', $validStatuses)
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Estado inválido. Estados válidos: ' . implode(', ', $validStatuses),
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Intentar actualizar
            $order->status = $request->status;
            $saved = $order->save();
            
            if (!$saved) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se pudo guardar el cambio en la base de datos'
                ], 500);
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Estado del pedido actualizado exitosamente',
                'new_status' => $order->status
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Pedido no encontrado'
            ], 404);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error de validación',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el estado: ' . $e->getMessage(),
                'error_detail' => $e->getTraceAsString()
            ], 500);
        }
    }

    /**
     * Eliminar un pedido
     */
    public function destroy($id)
    {
        try {
            $order = Order::findOrFail($id);
            
            // Eliminar items del pedido primero
            $order->orderItems()->delete();
            
            // Eliminar el pedido
            $order->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Pedido eliminado exitosamente'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el pedido'
            ], 500);
        }
    }

    /**
     * Obtener estadísticas de pedidos
     */
    public function getStats()
    {
        try {
            $stats = [
                'total' => Order::count(),
                'pendientes' => Order::where('status', 'pending')->count(),
                'confirmados' => Order::where('status', 'confirmed')->count(),
                'en_preparacion' => Order::where('status', 'processing')->count(),
                'enviados' => Order::where('status', 'shipped')->count(),
                'entregados' => Order::where('status', 'delivered')->count(),
                'cancelados' => Order::where('status', 'cancelled')->count(),
                'ventas_total' => Order::whereNotIn('status', ['cancelado'])->sum('total')
            ];
            
            return response()->json([
                'success' => true,
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar estadísticas'
            ], 500);
        }
    }

    /**
     * Obtener clase CSS para el estado
     */
    private function getStatusClass($status)
    {
        $classes = [
            'pendiente' => 'badge-warning',
            'confirmado' => 'badge-info',
            'en_preparacion' => 'badge-primary',
            'enviado' => 'badge-secondary',
            'entregado' => 'badge-success',
            'cancelado' => 'badge-danger'
        ];

        return $classes[$status] ?? 'badge-light';
    }
}
