<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Log;

class ProductController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Mostrar la vista principal de productos
     */
    public function index()
    {
        return view('administrador.products.index');
    }

    /**
     * Obtener datos de productos para AJAX
     */
    public function getData()
    {
        try {
            $products = Product::with('category')
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($product) {
                    return [
                        'id' => $product->id,
                        'name' => $product->name,
                        'category' => $product->category ? $product->category->name : 'Sin categoría',
                        'category_id' => $product->category_id,
                        'price' => number_format($product->price, 2),
                        'price_raw' => $product->price,
                        'image' => $product->image,
                        'description' => $product->description,
                        'created_at' => $product->created_at->format('d/m/Y H:i'),
                        'updated_at' => $product->updated_at->format('d/m/Y H:i')
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $products
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar productos: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mostrar un producto específico
     */
    public function show($id)
    {
        try {
            $product = Product::with('category')->findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $product->id,
                    'name' => $product->name,
                    'category' => $product->category,
                    'category_id' => $product->category_id,
                    'price' => $product->price,
                    'image' => $product->image,
                    'description' => $product->description,
                    'created_at' => $product->created_at->format('d/m/Y H:i'),
                    'updated_at' => $product->updated_at->format('d/m/Y H:i')
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Producto no encontrado'
            ], 404);
        }
    }

    /**
     * Crear un nuevo producto
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255|unique:products,name',
                'category_id' => 'required|exists:categories,id',
                'price' => 'required|numeric|min:0',
                'image' => 'nullable|url|max:500',
                'description' => 'nullable|string|max:1000'
            ], [
                'name.required' => 'El nombre del producto es obligatorio',
                'name.unique' => 'Ya existe un producto con este nombre',
                'category_id.required' => 'La categoría es obligatoria',
                'category_id.exists' => 'La categoría seleccionada no existe',
                'price.required' => 'El precio es obligatorio',
                'price.numeric' => 'El precio debe ser un número válido',
                'price.min' => 'El precio no puede ser negativo',
                'image.url' => 'La URL de la imagen no es válida'
            ]);

            $product = Product::create($validated);

            return response()->json([
                'success' => true,
                'message' => 'Producto creado exitosamente',
                'data' => $product
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error de validación',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al crear el producto: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar un producto existente
     */
    public function update(Request $request, $id)
    {
        try {
            $product = Product::findOrFail($id);

            $validated = $request->validate([
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    Rule::unique('products', 'name')->ignore($id)
                ],
                'category_id' => 'required|exists:categories,id',
                'price' => 'required|numeric|min:0',
                'image' => 'nullable|url|max:500',
                'description' => 'nullable|string|max:1000'
            ], [
                'name.required' => 'El nombre del producto es obligatorio',
                'name.unique' => 'Ya existe otro producto con este nombre',
                'category_id.required' => 'La categoría es obligatoria',
                'category_id.exists' => 'La categoría seleccionada no existe',
                'price.required' => 'El precio es obligatorio',
                'price.numeric' => 'El precio debe ser un número válido',
                'price.min' => 'El precio no puede ser negativo',
                'image.url' => 'La URL de la imagen no es válida'
            ]);

            $product->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Producto actualizado exitosamente',
                'data' => $product
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error de validación',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el producto: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Eliminar un producto
     */
    public function destroy($id)
    {
        try {
            $product = Product::findOrFail($id);

            // Verificar si el producto tiene pedidos asociados
            if ($product->orderItems && $product->orderItems->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se puede eliminar el producto porque tiene pedidos asociados'
                ], 400);
            }

            $productName = $product->name;
            $product->delete();

            return response()->json([
                'success' => true,
                'message' => "Producto '{$productName}' eliminado exitosamente"
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el producto: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener categorías para select
     */
    public function getCategories()
    {
        try {
            $categories = Category::orderBy('name')->get(['id', 'name']);
            
            return response()->json([
                'success' => true,
                'data' => $categories
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar categorías'
            ], 500);
        }
    }
}
