<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Quote;
use Illuminate\Http\Request;

class QuoteController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Mostrar la vista principal de cotizaciones
     */
    public function index()
    {
        return view('administrador.quotes.index');
    }

    /**
     * Obtener datos de cotizaciones para AJAX
     */
    public function getData()
    {
        try {
            $quotes = Quote::orderBy('created_at', 'desc')->get();
            
            return response()->json([
                'success' => true,
                'data' => $quotes->map(function($quote) {
                    return [
                        'id' => $quote->id,
                        'name' => $quote->name,
                        'email' => $quote->email,
                        'phone' => $quote->phone,
                        'product_name' => $quote->product_name,
                        'product_category' => $quote->product_category,
                        'quantity' => $quote->quantity,
                        'budget_range' => $quote->budget_range,
                        'message' => $quote->message,
                        'status' => $quote->status,
                        'status_name' => $quote->status_name,
                        'status_class' => $quote->status_class,
                        'priority' => $quote->priority,
                        'priority_class' => $quote->priority_class,
                        'created_at' => $quote->created_at->format('d/m/Y H:i'),
                        'created_at_human' => $quote->created_at->diffForHumans(),
                    ];
                })
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar las cotizaciones: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mostrar una cotización específica
     */
    public function show($id)
    {
        try {
            $quote = Quote::findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $quote->id,
                    'name' => $quote->name,
                    'email' => $quote->email,
                    'phone' => $quote->phone,
                    'product_name' => $quote->product_name,
                    'product_category' => $quote->product_category,
                    'quantity' => $quote->quantity,
                    'budget_range' => $quote->budget_range,
                    'message' => $quote->message,
                    'status' => $quote->status,
                    'status_name' => $quote->status_name,
                    'status_class' => $quote->status_class,
                    'priority' => $quote->priority,
                    'priority_class' => $quote->priority_class,
                    'created_at' => $quote->created_at->format('d/m/Y H:i'),
                    'created_at_human' => $quote->created_at->diffForHumans(),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Cotización no encontrada'
            ], 404);
        }
    }

    /**
     * Eliminar una cotización
     */
    public function destroy($id)
    {
        try {
            $quote = Quote::findOrFail($id);
            
            $quote->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Cotización eliminada exitosamente'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar la cotización: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar el estado de una cotización
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            $quote = Quote::findOrFail($id);
            
            $request->validate([
                'status' => 'required|in:pendiente,procesando,enviado,cerrado'
            ]);
            
            $quote->status = $request->status;
            $quote->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Estado actualizado exitosamente',
                'data' => [
                    'status' => $quote->status,
                    'status_name' => $quote->status_name,
                    'status_class' => $quote->status_class
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el estado: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar la prioridad de una cotización
     */
    public function updatePriority(Request $request, $id)
    {
        try {
            $quote = Quote::findOrFail($id);
            
            $request->validate([
                'priority' => 'required|in:alta,media,baja'
            ]);
            
            $quote->priority = $request->priority;
            $quote->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Prioridad actualizada exitosamente',
                'data' => [
                    'priority' => $quote->priority,
                    'priority_class' => $quote->priority_class
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar la prioridad: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener estadísticas de cotizaciones
     */
    public function getStats()
    {
        try {
            $stats = [
                'total' => Quote::count(),
                'pendientes' => Quote::where('status', 'pendiente')->count(),
                'procesando' => Quote::where('status', 'procesando')->count(),
                'enviados' => Quote::where('status', 'enviado')->count(),
                'cerrados' => Quote::where('status', 'cerrado')->count(),
                'alta_prioridad' => Quote::where('priority', 'alta')->count(),
                'media_prioridad' => Quote::where('priority', 'media')->count(),
                'baja_prioridad' => Quote::where('priority', 'baja')->count(),
            ];
            
            return response()->json([
                'success' => true,
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar estadísticas: ' . $e->getMessage()
            ], 500);
        }
    }
}
