<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\ContactMessage;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Response;

class ReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Mostrar la vista principal de reportes
     */
    public function index()
    {
        return view('administrador.reports.index');
    }

    /**
     * Reporte de pedidos
     */
    public function ordersReport(Request $request)
    {
        $query = Order::with(['user', 'orderItems']);
        
        // Filtro por fecha
        if ($request->filled('fecha_inicio')) {
            $query->whereDate('created_at', '>=', $request->fecha_inicio);
        }
        
        if ($request->filled('fecha_fin')) {
            $query->whereDate('created_at', '<=', $request->fecha_fin);
        }
        
        // Filtro por estado
        if ($request->filled('estado')) {
            $query->where('status', $request->estado);
        }
        
        $orders = $query->orderBy('created_at', 'desc')->get();
        
        if ($request->input('formato') === 'excel') {
            return $this->downloadOrdersExcel($orders);
        }
        
        return response()->json([
            'success' => true,
            'data' => $orders->map(function($order) {
                return [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'cliente' => $order->full_name,
                    'email' => $order->user ? $order->user->email : 'N/A',
                    'telefono' => $order->phone,
                    'direccion' => $order->address . ', ' . $order->district,
                    'metodo_pago' => $order->payment_method,
                    'subtotal' => number_format($order->subtotal, 2),
                    'envio' => number_format($order->shipping_cost, 2),
                    'total' => number_format($order->total, 2),
                    'items' => $order->orderItems->count(),
                    'estado' => $this->getStatusText($order->status),
                    'fecha' => $order->created_at->format('d/m/Y H:i')
                ];
            })
        ]);
    }

    /**
     * Reporte de mensajes
     */
    public function messagesReport(Request $request)
    {
        $query = ContactMessage::query();
        
        // Filtro por estado
        if ($request->filled('estado')) {
            $query->where('status', $request->estado);
        }
        
        // Filtro por fecha
        if ($request->filled('fecha_inicio')) {
            $query->whereDate('created_at', '>=', $request->fecha_inicio);
        }
        
        if ($request->filled('fecha_fin')) {
            $query->whereDate('created_at', '<=', $request->fecha_fin);
        }
        
        $messages = $query->orderBy('created_at', 'desc')->get();
        
        if ($request->input('formato') === 'excel') {
            return $this->downloadMessagesExcel($messages);
        }
        
        return response()->json([
            'success' => true,
            'data' => $messages->map(function($message) {
                return [
                    'id' => $message->id,
                    'nombre' => $message->name,
                    'email' => $message->email,
                    'telefono' => $message->phone ?: 'N/A',
                    'asunto' => $message->subject_name,
                    'mensaje' => substr($message->message, 0, 100) . '...',
                    'estado' => $this->getMessageStatusText($message->status),
                    'fecha' => $message->created_at->format('d/m/Y H:i')
                ];
            })
        ]);
    }

    /**
     * Reporte de productos
     */
    public function productsReport(Request $request)
    {
        $query = Product::with('category');
        
        // Filtro por categoría
        if ($request->filled('categoria')) {
            $query->where('category_id', $request->categoria);
        }
        
        $products = $query->orderBy('name')->get();
        
        if ($request->input('formato') === 'excel') {
            return $this->downloadProductsExcel($products);
        }
        
        return response()->json([
            'success' => true,
            'data' => $products->map(function($product) {
                return [
                    'id' => $product->id,
                    'nombre' => $product->name,
                    'categoria' => $product->category ? $product->category->name : 'Sin categoría',
                    'precio' => 'S/ ' . number_format($product->price, 2),
                    'stock' => $product->stock ?? 'N/A',
                    'estado' => $product->status ?? 'activo',
                    'fecha' => $product->created_at->format('d/m/Y')
                ];
            })
        ]);
    }

    /**
     * Reporte de usuarios
     */
    public function usersReport(Request $request)
    {
        $query = User::where('role', 'customer');
        
        // Filtro por fecha de registro
        if ($request->filled('fecha_inicio')) {
            $query->whereDate('created_at', '>=', $request->fecha_inicio);
        }
        
        if ($request->filled('fecha_fin')) {
            $query->whereDate('created_at', '<=', $request->fecha_fin);
        }
        
        $users = $query->orderBy('created_at', 'desc')->get();
        
        if ($request->input('formato') === 'excel') {
            return $this->downloadUsersExcel($users);
        }
        
        return response()->json([
            'success' => true,
            'data' => $users->map(function($user) {
                return [
                    'id' => $user->id,
                    'nombre' => $user->name,
                    'email' => $user->email,
                    'pedidos' => $user->orders ? $user->orders->count() : 0,
                    'fecha_registro' => $user->created_at->format('d/m/Y H:i'),
                    'ultimo_acceso' => $user->last_login_at ? $user->last_login_at->format('d/m/Y H:i') : 'Nunca'
                ];
            })
        ]);
    }

    /**
     * Descargar reporte de pedidos en Excel (CSV)
     */
    private function downloadOrdersExcel($orders)
    {
        $filename = 'reporte_pedidos_' . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($orders) {
            $file = fopen('php://output', 'w');
            
            // BOM para UTF-8
            fwrite($file, "\xEF\xBB\xBF");
            
            // Encabezados
            fputcsv($file, [
                'ID', 'Número de Pedido', 'Cliente', 'Email', 'Teléfono',
                'Dirección', 'Método de Pago', 'Subtotal', 'Envío', 'Total',
                'Items', 'Estado', 'Fecha'
            ]);
            
            // Datos
            foreach ($orders as $order) {
                fputcsv($file, [
                    $order->id,
                    $order->order_number,
                    $order->full_name,
                    $order->user ? $order->user->email : 'N/A',
                    $order->phone,
                    $order->address . ', ' . $order->district,
                    ucfirst($order->payment_method),
                    'S/ ' . number_format($order->subtotal, 2),
                    'S/ ' . number_format($order->shipping_cost, 2),
                    'S/ ' . number_format($order->total, 2),
                    $order->orderItems->count(),
                    $this->getStatusText($order->status),
                    $order->created_at->format('d/m/Y H:i')
                ]);
            }
            
            fclose($file);
        };
        
        return Response::stream($callback, 200, $headers);
    }

    /**
     * Descargar reporte de mensajes en Excel (CSV)
     */
    private function downloadMessagesExcel($messages)
    {
        $filename = 'reporte_mensajes_' . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($messages) {
            $file = fopen('php://output', 'w');
            
            // BOM para UTF-8
            fwrite($file, "\xEF\xBB\xBF");
            
            // Encabezados
            fputcsv($file, [
                'ID', 'Nombre', 'Email', 'Teléfono', 'Asunto', 'Mensaje', 'Estado', 'Fecha'
            ]);
            
            // Datos
            foreach ($messages as $message) {
                fputcsv($file, [
                    $message->id,
                    $message->name,
                    $message->email,
                    $message->phone ?: 'N/A',
                    $message->subject_name,
                    $message->message,
                    $this->getMessageStatusText($message->status),
                    $message->created_at->format('d/m/Y H:i')
                ]);
            }
            
            fclose($file);
        };
        
        return Response::stream($callback, 200, $headers);
    }

    /**
     * Descargar reporte de productos en Excel (CSV)
     */
    private function downloadProductsExcel($products)
    {
        $filename = 'reporte_productos_' . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($products) {
            $file = fopen('php://output', 'w');
            
            // BOM para UTF-8
            fwrite($file, "\xEF\xBB\xBF");
            
            // Encabezados
            fputcsv($file, [
                'ID', 'Nombre', 'Categoría', 'Precio', 'Stock', 'Estado', 'Fecha Creación'
            ]);
            
            // Datos
            foreach ($products as $product) {
                fputcsv($file, [
                    $product->id,
                    $product->name,
                    $product->category ? $product->category->name : 'Sin categoría',
                    'S/ ' . number_format($product->price, 2),
                    $product->stock ?? 'N/A',
                    $product->status ?? 'activo',
                    $product->created_at->format('d/m/Y H:i')
                ]);
            }
            
            fclose($file);
        };
        
        return Response::stream($callback, 200, $headers);
    }

    /**
     * Descargar reporte de usuarios en Excel (CSV)
     */
    private function downloadUsersExcel($users)
    {
        $filename = 'reporte_usuarios_' . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($users) {
            $file = fopen('php://output', 'w');
            
            // BOM para UTF-8
            fwrite($file, "\xEF\xBB\xBF");
            
            // Encabezados
            fputcsv($file, [
                'ID', 'Nombre', 'Email', 'Total Pedidos', 'Fecha Registro', 'Último Acceso'
            ]);
            
            // Datos
            foreach ($users as $user) {
                fputcsv($file, [
                    $user->id,
                    $user->name,
                    $user->email,
                    $user->orders ? $user->orders->count() : 0,
                    $user->created_at->format('d/m/Y H:i'),
                    $user->last_login_at ? $user->last_login_at->format('d/m/Y H:i') : 'Nunca'
                ]);
            }
            
            fclose($file);
        };
        
        return Response::stream($callback, 200, $headers);
    }

    private function getStatusText($status)
    {
        $statusTexts = [
            'pending' => 'Pendiente',
            'confirmed' => 'Confirmado',
            'processing' => 'En Preparación',
            'shipped' => 'Enviado',
            'delivered' => 'Entregado',
            'cancelled' => 'Cancelado'
        ];
        return $statusTexts[$status] ?? $status;
    }

    private function getMessageStatusText($status)
    {
        $statusTexts = [
            'nuevo' => 'Nuevo',
            'leido' => 'Leído',
            'respondido' => 'Respondido',
            'cerrado' => 'Cerrado'
        ];
        return $statusTexts[$status] ?? $status;
    }
}
