<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Mostrar lista de administradores
     */
    public function indexAdmins()
    {
        return view('administrador.users.admins');
    }

    /**
     * Mostrar lista de clientes
     */
    public function indexClients()
    {
        return view('administrador.users.clients');
    }

    /**
     * Obtener administradores para AJAX
     */
    public function getAdminsData()
    {
        try {
            $admins = User::admins()
                ->withCount('orders')
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'phone' => $user->phone ?? 'Sin teléfono',
                        'orders_count' => $user->orders_count ?? 0,
                        'created_at' => $user->created_at->format('d/m/Y H:i'),
                        'updated_at' => $user->updated_at->format('d/m/Y H:i')
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $admins
            ]);
        } catch (\Exception $e) {
            Log::error('Error al cargar administradores: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar los administradores: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener clientes para AJAX
     */
    public function getClientsData()
    {
        try {
            $clients = User::where('rol', '!=', 'administrador')
                ->withCount('orders')
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'phone' => $user->phone ?? 'Sin teléfono',
                        'rol' => $user->rol ?? 'cliente',
                        'orders_count' => $user->orders_count ?? 0,
                        'created_at' => $user->created_at->format('d/m/Y H:i'),
                        'updated_at' => $user->updated_at->format('d/m/Y H:i')
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $clients
            ]);
        } catch (\Exception $e) {
            Log::error('Error al cargar clientes: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar los clientes: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mostrar un usuario específico
     */
    public function show($id)
    {
        try {
            $user = User::withCount('orders')->findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'rol' => $user->rol,
                    'orders_count' => $user->orders_count ?? 0,
                    'created_at' => $user->created_at->format('d/m/Y H:i'),
                    'updated_at' => $user->updated_at->format('d/m/Y H:i')
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado'
            ], 404);
        }
    }

    /**
     * Crear un nuevo usuario
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email',
                'password' => 'required|string|min:6',
                'phone' => 'nullable|string|max:20',
                'rol' => 'required|in:cliente,administrador'
            ], [
                'name.required' => 'El nombre es obligatorio',
                'email.required' => 'El email es obligatorio',
                'email.email' => 'El email debe ser válido',
                'email.unique' => 'Ya existe un usuario con este email',
                'password.required' => 'La contraseña es obligatoria',
                'password.min' => 'La contraseña debe tener al menos 6 caracteres',
                'rol.required' => 'El rol es obligatorio',
                'rol.in' => 'El rol debe ser cliente o administrador'
            ]);

            $user = User::create($validated);

            return response()->json([
                'success' => true,
                'message' => 'Usuario creado exitosamente',
                'data' => $user
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error de validación',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Error al crear usuario: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al crear el usuario: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar un usuario existente
     */
    public function update(Request $request, $id)
    {
        try {
            $user = User::findOrFail($id);

            $rules = [
                'name' => 'required|string|max:255',
                'email' => [
                    'required',
                    'email',
                    Rule::unique('users', 'email')->ignore($id)
                ],
                'phone' => 'nullable|string|max:20',
                'rol' => 'required|in:cliente,administrador'
            ];

            // Solo validar contraseña si se proporciona
            if ($request->filled('password')) {
                $rules['password'] = 'string|min:6';
            }

            $validated = $request->validate($rules, [
                'name.required' => 'El nombre es obligatorio',
                'email.required' => 'El email es obligatorio',
                'email.email' => 'El email debe ser válido',
                'email.unique' => 'Ya existe otro usuario con este email',
                'password.min' => 'La contraseña debe tener al menos 6 caracteres',
                'rol.required' => 'El rol es obligatorio',
                'rol.in' => 'El rol debe ser cliente o administrador'
            ]);

            // Si no se proporciona contraseña, no la actualizamos
            if (!$request->filled('password')) {
                unset($validated['password']);
            }

            $user->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Usuario actualizado exitosamente',
                'data' => $user
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error de validación',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Error al actualizar usuario: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el usuario: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Eliminar un usuario
     */
    public function destroy($id)
    {
        try {
            $user = User::findOrFail($id);

            // Verificar si el usuario tiene pedidos asociados
            if ($user->orders && $user->orders->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se puede eliminar el usuario porque tiene pedidos asociados'
                ], 400);
            }

            // No permitir eliminar el último administrador
            if ($user->isAdmin()) {
                $adminCount = User::admins()->count();
                if ($adminCount <= 1) {
                    return response()->json([
                        'success' => false,
                        'message' => 'No se puede eliminar el último administrador del sistema'
                    ], 400);
                }
            }

            $userName = $user->name;
            $user->delete();

            return response()->json([
                'success' => true,
                'message' => "Usuario '{$userName}' eliminado exitosamente"
            ]);
        } catch (\Exception $e) {
            Log::error('Error al eliminar usuario: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el usuario: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Cambiar contraseña de un usuario
     */
    public function changePassword(Request $request, $id)
    {
        try {
            $user = User::findOrFail($id);

            $validated = $request->validate([
                'password' => 'required|string|min:6|confirmed'
            ], [
                'password.required' => 'La nueva contraseña es obligatoria',
                'password.min' => 'La contraseña debe tener al menos 6 caracteres',
                'password.confirmed' => 'Las contraseñas no coinciden'
            ]);

            $user->update(['password' => $validated['password']]);

            return response()->json([
                'success' => true,
                'message' => 'Contraseña actualizada exitosamente'
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error de validación',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Error al cambiar contraseña: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error al cambiar la contraseña'
            ], 500);
        }
    }
}
