<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use App\Models\User;

class AuthController extends Controller
{
    public function showLogin()
    {
        return view('auth.login');
    }

    public function showRegister()
    {
        return view('auth.register');
    }

    public function register(Request $request)
    {
        // Validar datos
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email',
            'password' => 'required|string|min:6',
            'password_confirmation' => 'required|same:password',
            'phone' => 'nullable|string|max:20'
        ]);

        try {
            // Crear usuario
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'phone' => $request->phone,
                'rol' => 'cliente'
            ]);

            // Auto login después de registro
            Session::put('user_id', $user->id);
            Session::put('user_name', $user->name);
            Session::put('user_email', $user->email);

            return response()->json([
                'success' => true,
                'message' => 'Cuenta creada exitosamente',
                'redirect' => route('Producto.index')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al crear la cuenta: ' . $e->getMessage()
            ]);
        }
    }

    public function login(Request $request)
    {
        $email = $request->email;
        $password = $request->password;

        // Buscar usuario usando el modelo
        $user = User::where('email', $email)->first();

        if ($user && Hash::check($password, $user->password)) {
            // Login exitoso
            Session::put('user_id', $user->id);
            Session::put('user_name', $user->name);
            Session::put('user_email', $user->email);
            
            return response()->json([
                'success' => true,
                'message' => 'Login exitoso',
                'redirect' => route('Producto.index')
            ]);
        } else {
            // Login fallido
            return response()->json([
                'success' => false,
                'message' => 'Credenciales incorrectas'
            ]);
        }
    }

    public function logout()
    {
        Session::flush();
        return redirect()->route('home');
    }

    public function checkAuth()
    {
        return response()->json([
            'logged_in' => Session::has('user_id')
        ]);
    }

    public function createTestUser()
    {
        $email = 'test@electromart.com';
        $password = 'password';
        
        // Verificar si el usuario ya existe usando el modelo
        $existingUser = User::where('email', $email)->first();
        
        if (!$existingUser) {
            User::create([
                'name' => 'Usuario Test',
                'email' => $email,
                'password' => Hash::make($password)
            ]);
        }
        
        return response()->json([
            'success' => true,
            'email' => $email,
            'password' => $password
        ]);
    }
}
