<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\BlogPost;

class BlogController extends Controller
{
    // Mostrar todos los posts del blog
    public function index(Request $request)
    {
        $query = BlogPost::published();

        // Filtrar por categoría si se proporciona
        if ($request->has('category') && $request->category != '') {
            $query->byCategory($request->category);
        }

        // Buscar por título o contenido
        if ($request->has('search') && $request->search != '') {
            $query->where(function($q) use ($request) {
                $q->where('title', 'LIKE', '%' . $request->search . '%')
                  ->orWhere('excerpt', 'LIKE', '%' . $request->search . '%');
            });
        }

        $posts = $query->orderByDesc('published_at')->paginate(12);
        $categories = BlogPost::getCategories();
        $featuredPosts = BlogPost::published()->featured()->recent(3)->get();

        return view('blog.index', compact('posts', 'categories', 'featuredPosts'));
    }

    // Mostrar un post específico
    public function show($slug)
    {
        $post = BlogPost::where('slug', $slug)->published()->firstOrFail();
        
        // Incrementar vistas
        $post->incrementViews();
        
        // Obtener posts relacionados
        $relatedPosts = $post->getRelatedPosts(3);
        
        // Obtener posts recientes
        $recentPosts = BlogPost::published()->recent(5)->get();

        return view('blog.show', compact('post', 'relatedPosts', 'recentPosts'));
    }

    // API para obtener posts destacados (para la home)
    public function getFeaturedPosts()
    {
        $posts = BlogPost::getHomePagePosts(3);
        
        return response()->json([
            'success' => true,
            'posts' => $posts->map(function($post) {
                return [
                    'id' => $post->id,
                    'title' => $post->title,
                    'excerpt' => $post->short_excerpt,
                    'image_url' => $post->image_url,
                    'category' => $post->category,
                    'author' => $post->author,
                    'date' => $post->spanish_date,
                    'url' => $post->url,
                    'read_time' => $post->read_time
                ];
            })
        ]);
    }

    // Admin: Ver todos los posts
    public function adminIndex()
    {
        return view('administrador.blog.index');
    }

    // Admin: Crear nuevo post
    public function create()
    {
        return view('administrador.blog.create');
    }

    // Admin: Guardar nuevo post
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'category' => 'required|string|max:100',
                'author' => 'required|string|max:255',
                'content' => 'required|string',
                'status' => 'required|in:borrador,publicado,programado',
                'image_url' => 'nullable|url|max:500',
                'excerpt' => 'nullable|string|max:500',
                'published_at' => 'nullable|date',
                'meta_title' => 'nullable|string|max:255',
                'meta_description' => 'nullable|string|max:500',
                'featured' => 'boolean'
            ]);

            // Generar slug automáticamente
            $validated['slug'] = \Str::slug($validated['title']);

            // Establecer autor por defecto si no se proporciona
            if (empty($validated['author'])) {
                $validated['author'] = 'ElectroMart';
            }

            // Establecer fecha de publicación si el estado es publicado y no se especifica fecha
            if ($validated['status'] === 'publicado' && empty($validated['published_at'])) {
                $validated['published_at'] = now();
            }

            $post = BlogPost::create($validated);

            return response()->json([
                'success' => true,
                'message' => '¡Post creado correctamente!',
                'post' => $post
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Por favor verifica los datos ingresados',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al crear el post'
            ], 500);
        }
    }

    // Admin: Editar post
    public function edit($id)
    {
        $post = BlogPost::findOrFail($id);
        return view('admin.blog.edit', compact('post'));
    }

    // Admin: Actualizar post
    public function update(Request $request, $id)
    {
        try {
            $post = BlogPost::findOrFail($id);
            
            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'category' => 'required|string|max:100',
                'author' => 'required|string|max:255',
                'content' => 'required|string',
                'status' => 'required|in:borrador,publicado,programado',
                'image_url' => 'nullable|url|max:500',
                'excerpt' => 'nullable|string|max:500',
                'published_at' => 'nullable|date',
                'meta_title' => 'nullable|string|max:255',
                'meta_description' => 'nullable|string|max:500',
                'featured' => 'boolean'
            ]);

            // Actualizar slug basado en el título
            $validated['slug'] = \Str::slug($validated['title']);

            // Establecer fecha de publicación si cambia a publicado y no se especifica fecha
            if ($validated['status'] === 'publicado' && $post->status !== 'publicado' && empty($validated['published_at'])) {
                $validated['published_at'] = now();
            }

            $post->update($validated);

            return response()->json([
                'success' => true,
                'message' => '¡Post actualizado correctamente!'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el post'
            ], 500);
        }
    }

    // Admin: Eliminar post
    public function destroy($id)
    {
        try {
            $post = BlogPost::findOrFail($id);
            $post->delete();

            return response()->json([
                'success' => true,
                'message' => 'Post eliminado correctamente'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el post'
            ], 500);
        }
    }

    /**
     * Obtener datos de posts para AJAX (administración)
     */
    public function getData()
    {
        try {
            $posts = BlogPost::orderBy('created_at', 'desc')->get();
            
            return response()->json([
                'success' => true,
                'data' => $posts->map(function($post) {
                    return [
                        'id' => $post->id,
                        'title' => $post->title,
                        'category' => $post->category,
                        'author' => $post->author,
                        'status' => $post->status,
                        'featured' => $post->featured,
                        'excerpt' => $post->excerpt,
                        'views_count' => $post->views_count ?? 0,
                        'published_at' => $post->published_at ? $post->published_at->format('Y-m-d H:i') : null,
                        'created_at' => $post->created_at->format('Y-m-d H:i')
                    ];
                })
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar los posts'
            ], 500);
        }
    }

    /**
     * Obtener categorías únicas para filtros
     */
    public function getCategories()
    {
        try {
            $categories = BlogPost::distinct('category')->pluck('category')->filter();
            
            return response()->json([
                'success' => true,
                'data' => $categories->values()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar las categorías'
            ], 500);
        }
    }

    /**
     * Mostrar un post específico para edición (administración)
     */
    public function showForEdit($id)
    {
        try {
            $post = BlogPost::findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $post->id,
                    'title' => $post->title,
                    'category' => $post->category,
                    'author' => $post->author,
                    'status' => $post->status,
                    'featured' => $post->featured,
                    'image_url' => $post->image_url,
                    'excerpt' => $post->excerpt,
                    'content' => $post->content,
                    'published_at' => $post->published_at ? $post->published_at->format('Y-m-d\TH:i') : null,
                    'meta_title' => $post->meta_title,
                    'meta_description' => $post->meta_description
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Post no encontrado'
            ], 404);
        }
    }

    /**
     * Cambiar estado destacado de un post
     */
    public function toggleFeatured($id)
    {
        try {
            $post = BlogPost::findOrFail($id);
            $post->featured = !$post->featured;
            $post->save();
            
            return response()->json([
                'success' => true,
                'message' => $post->featured ? 'Post marcado como destacado' : 'Post desmarcado como destacado'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cambiar el estado destacado'
            ], 500);
        }
    }
}
