<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\ShoppingCart;
use App\Models\CartItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CartController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function getCart()
    {
        $user = Auth::user();
        $cart = $this->getOrCreateCart($user);
        
        $cartItems = $cart->items()->with(['product.images', 'product.brand'])->get();
        
        $cartData = [
            'id' => $cart->id,
            'items' => $cartItems->map(function($item) {
                return [
                    'id' => $item->id,
                    'product_id' => $item->product_id,
                    'product_name' => $item->product->name,
                    'product_image' => $item->product->images->first()->image_url ?? '/images/default-product.jpg',
                    'brand' => $item->product->brand->name ?? '',
                    'price' => $item->unit_price,
                    'quantity' => $item->quantity,
                    'subtotal' => $item->total_price,
                    'product_stock' => $item->product->stock_quantity
                ];
            }),
            'total_items' => $cart->total_items,
            'total_amount' => $cart->total_amount,
            'subtotal' => $cart->subtotal,
            'tax_amount' => $cart->tax_amount,
            'discount_amount' => $cart->discount_amount ?? 0
        ];

        return response()->json([
            'success' => true,
            'cart' => $cartData
        ]);
    }

    public function addToCart(Request $request)
    {
        \Log::info('CartController addToCart called', $request->all());
        
        try {
            $request->validate([
                'product_id' => 'required|exists:products,id',
                'quantity' => 'required|integer|min:1'
            ]);

            $user = Auth::user();
            \Log::info('User authenticated', ['user_id' => $user->id]);
            
            $product = Product::findOrFail($request->product_id);
            \Log::info('Product found', ['product_id' => $product->id, 'name' => $product->name]);
            
            // Verificar stock
            if ($product->stock_quantity < $request->quantity) {
                return response()->json([
                    'success' => false,
                    'message' => 'Stock insuficiente. Solo quedan ' . $product->stock_quantity . ' unidades disponibles.'
                ], 400);
            }

            $cart = $this->getOrCreateCart($user);
            \Log::info('Cart obtained', ['cart_id' => $cart->id]);
            
            // Verificar si el producto ya está en el carrito
            $existingItem = $cart->items()->where('product_id', $request->product_id)->first();
            
            if ($existingItem) {
                $newQuantity = $existingItem->quantity + $request->quantity;
                
                if ($product->stock_quantity < $newQuantity) {
                    return response()->json([
                        'success' => false,
                        'message' => 'No se puede agregar esa cantidad. Stock máximo disponible: ' . $product->stock_quantity
                    ], 400);
                }
                
                $existingItem->update([
                    'quantity' => $newQuantity,
                    'total_price' => $newQuantity * $existingItem->unit_price
                ]);
                
                $message = 'Cantidad actualizada en el carrito';
            } else {
                $cartItem = CartItem::create([
                    'cart_id' => $cart->id,
                    'product_id' => $request->product_id,
                    'quantity' => $request->quantity,
                    'unit_price' => $product->price,
                    'total_price' => $request->quantity * $product->price
                ]);
                \Log::info('Cart item created', ['cart_item_id' => $cartItem->id]);
                
                $message = 'Producto agregado al carrito';
            }

            $this->updateCartTotals($cart);

            return response()->json([
                'success' => true,
                'message' => $message,
                'cart_count' => $cart->fresh()->total_items
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error in addToCart', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'success' => false,
                'message' => 'Error interno del servidor: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateQuantity(Request $request)
    {
        $request->validate([
            'item_id' => 'required|exists:cart_items,id',
            'quantity' => 'required|integer|min:1'
        ]);

        $user = Auth::user();
        $cartItem = CartItem::findOrFail($request->item_id);
        
        // Verificar que el item pertenece al carrito del usuario
        if ($cartItem->cart->user_id !== $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'Item no encontrado'
            ], 404);
        }

        $product = $cartItem->product;
        
        if ($product->stock_quantity < $request->quantity) {
            return response()->json([
                'success' => false,
                'message' => 'Stock insuficiente. Solo quedan ' . $product->stock_quantity . ' unidades disponibles.'
            ], 400);
        }

        $cartItem->update([
            'quantity' => $request->quantity,
            'total_price' => $request->quantity * $cartItem->unit_price
        ]);

        $this->updateCartTotals($cartItem->cart);

        return response()->json([
            'success' => true,
            'message' => 'Cantidad actualizada',
            'new_subtotal' => $cartItem->total_price,
            'cart_total' => $cartItem->cart->fresh()->total_amount
        ]);
    }

    public function removeItem(Request $request)
    {
        $request->validate([
            'item_id' => 'required|exists:cart_items,id'
        ]);

        $user = Auth::user();
        $cartItem = CartItem::findOrFail($request->item_id);
        
        // Verificar que el item pertenece al carrito del usuario
        if ($cartItem->cart->user_id !== $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'Item no encontrado'
            ], 404);
        }

        $cart = $cartItem->cart;
        $cartItem->delete();
        
        $this->updateCartTotals($cart);

        return response()->json([
            'success' => true,
            'message' => 'Producto eliminado del carrito',
            'cart_count' => $cart->fresh()->total_items,
            'cart_total' => $cart->fresh()->total_amount
        ]);
    }

    public function clearCart()
    {
        $user = Auth::user();
        $cart = $user->getActiveCart();
        
        if ($cart) {
            $cart->items()->delete();
            $this->updateCartTotals($cart);
        }

        return response()->json([
            'success' => true,
            'message' => 'Carrito vaciado'
        ]);
    }

    public function getCartCount()
    {
        if (!Auth::check()) {
            return response()->json(['count' => 0]);
        }

        $user = Auth::user();
        $cart = $user->getActiveCart();
        
        return response()->json([
            'count' => $cart ? $cart->total_items : 0
        ]);
    }

    private function getOrCreateCart($user)
    {
        $cart = $user->getActiveCart();
        
        if (!$cart) {
            $cart = ShoppingCart::create([
                'user_id' => $user->id,
                'status' => 'active',
                'total_items' => 0,
                'subtotal' => 0,
                'tax_amount' => 0,
                'total_amount' => 0
            ]);
        }
        
        return $cart;
    }

    private function updateCartTotals($cart)
    {
        $items = $cart->items;
        
        $totalItems = $items->sum('quantity');
        $subtotal = $items->sum('total_price');
        $taxRate = 0.18; // 18% IGV
        $taxAmount = $subtotal * $taxRate;
        $totalAmount = $subtotal + $taxAmount;
        
        $cart->update([
            'total_items' => $totalItems,
            'subtotal' => $subtotal,
            'tax_amount' => $taxAmount,
            'total_amount' => $totalAmount
        ]);
    }
}
