<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ContactMessage;
use App\Models\Quote;
use App\Models\CompanyInfo;

class ContactController extends Controller
{
    // Enviar mensaje de contacto
    public function sendMessage(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'phone' => 'nullable|string|max:20',
                'subject' => 'required|in:ventas,servicio,garantia,otros',
                'message' => 'required|string|max:2000'
            ]);

            $contactMessage = ContactMessage::create($validated);

            return response()->json([
                'success' => true,
                'message' => '¡Mensaje enviado correctamente! Te responderemos pronto.',
                'data' => $contactMessage
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Por favor verifica los datos ingresados',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al enviar el mensaje. Inténtalo nuevamente.'
            ], 500);
        }
    }

    // Solicitar cotización
    public function requestQuote(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'phone' => 'required|string|max:20',
                'product_name' => 'nullable|string|max:255',
                'product_category' => 'nullable|string|max:100',
                'quantity' => 'nullable|integer|min:1|max:100',
                'budget_range' => 'nullable|string|max:100',
                'message' => 'nullable|string|max:2000'
            ]);

            // Determinar prioridad basada en el presupuesto
            $priority = 'media';
            if (isset($validated['budget_range'])) {
                if (strpos($validated['budget_range'], '10000') !== false || 
                    strpos($validated['budget_range'], 'más') !== false) {
                    $priority = 'alta';
                } elseif (strpos($validated['budget_range'], '1000') !== false) {
                    $priority = 'baja';
                }
            }

            $validated['priority'] = $priority;
            $quote = Quote::create($validated);

            return response()->json([
                'success' => true,
                'message' => '¡Cotización solicitada correctamente! Te contactaremos pronto.',
                'data' => $quote
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Por favor verifica los datos ingresados',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al solicitar cotización. Inténtalo nuevamente.'
            ], 500);
        }
    }

    // Obtener información de la empresa
    public function getCompanyInfo()
    {
        try {
            $generalInfo = CompanyInfo::getMainInfo();
            $salesInfo = CompanyInfo::getSalesInfo();
            $serviceInfo = CompanyInfo::getServiceInfo();

            return response()->json([
                'success' => true,
                'data' => [
                    'general' => $generalInfo,
                    'sales' => $salesInfo,
                    'service' => $serviceInfo
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener información de la empresa'
            ], 500);
        }
    }

    // Admin: Ver todos los mensajes
    public function indexMessages()
    {
        $messages = ContactMessage::orderBy('created_at', 'desc')->paginate(15);
        return view('admin.messages.index', compact('messages'));
    }

    // Admin: Ver todas las cotizaciones
    public function indexQuotes()
    {
        $quotes = Quote::orderBy('created_at', 'desc')->paginate(15);
        return view('admin.quotes.index', compact('quotes'));
    }

    // Admin: Marcar mensaje como leído
    public function markMessageAsRead($id)
    {
        try {
            $message = ContactMessage::findOrFail($id);
            $message->markAsRead();

            return response()->json([
                'success' => true,
                'message' => 'Mensaje marcado como leído'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el mensaje'
            ], 500);
        }
    }

    // Admin: Actualizar estado de cotización
    public function updateQuoteStatus(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'status' => 'required|in:pendiente,procesando,enviado,cerrado'
            ]);

            $quote = Quote::findOrFail($id);
            $quote->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Estado de cotización actualizado'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar la cotización'
            ], 500);
        }
    }
}
