<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;
use App\Models\Cart;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Payment;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;

class ProductoController extends Controller
{
    public function index(Request $request)
    {
        // Obtener productos de la base de datos con paginación
        $query = Product::with('category');
        
        // Aplicar filtros si vienen por AJAX
        if ($request->ajax()) {
            return $this->filterProducts($request);
        }
        
        $products = $query->paginate(12);
        $categories = Category::all();
        
        // Si no hay categorías, crear algunas por defecto para prueba
        if ($categories->isEmpty()) {
            $defaultCategories = [
                ['name' => 'Refrigeración', 'description' => 'Refrigeradoras y congeladores'],
                ['name' => 'Lavandería', 'description' => 'Lavadoras y secadoras'],
                ['name' => 'Cocina', 'description' => 'Cocinas, hornos y microondas'],
                ['name' => 'Climatización', 'description' => 'Aires acondicionados']
            ];
            
            foreach ($defaultCategories as $catData) {
                Category::firstOrCreate(['name' => $catData['name']], $catData);
            }
            
            $categories = Category::all();
        }
        
        return view('productos.index', compact('products', 'categories'));
    }

    public function filterProducts(Request $request)
    {
        $query = Product::with('category');
        
        // Filtro por categoría
        if ($request->has('category') && $request->category !== 'all') {
            $query->where('category_id', $request->category);
        }
        
        // Filtro por rango de precio
        if ($request->has('price_ranges') && is_array($request->price_ranges)) {
            $query->where(function($q) use ($request) {
                foreach ($request->price_ranges as $range) {
                    switch ($range) {
                        case 'under_500':
                            $q->orWhere('price', '<', 500);
                            break;
                        case '500_1000':
                            $q->orWhereBetween('price', [500, 1000]);
                            break;
                        case '1000_2000':
                            $q->orWhereBetween('price', [1000, 2000]);
                            break;
                        case 'over_2000':
                            $q->orWhere('price', '>', 2000);
                            break;
                    }
                }
            });
        }
        
        // Filtro por búsqueda de texto
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function($q) use ($searchTerm) {
                $q->where('name', 'LIKE', "%{$searchTerm}%")
                  ->orWhere('description', 'LIKE', "%{$searchTerm}%");
            });
        }
        
        // Ordenamiento
        if ($request->has('sort')) {
            switch ($request->sort) {
                case 'price_asc':
                    $query->orderBy('price', 'asc');
                    break;
                case 'price_desc':
                    $query->orderBy('price', 'desc');
                    break;
                case 'name_asc':
                    $query->orderBy('name', 'asc');
                    break;
                case 'name_desc':
                    $query->orderBy('name', 'desc');
                    break;
                default:
                    $query->orderBy('created_at', 'desc');
            }
        }
        
        // Paginación
        $perPage = $request->has('per_page') ? $request->per_page : 12;
        $products = $query->paginate($perPage);
        
        // Si es una petición AJAX, devolver JSON
        if ($request->ajax()) {
            $productsData = $products->items();
            
            // Agregar datos formateados para cada producto
            foreach ($productsData as $product) {
                $product->formatted_price = $product->getFormattedPriceAttribute();
            }
            
            return response()->json([
                'success' => true,
                'products' => $productsData,
                'pagination' => [
                    'current_page' => $products->currentPage(),
                    'last_page' => $products->lastPage(),
                    'per_page' => $products->perPage(),
                    'total' => $products->total(),
                    'from' => $products->firstItem(),
                    'to' => $products->lastItem()
                ]
            ]);
        }
        
        return view('productos.index', compact('products'));
    }

    public function addToCart(Request $request)
    {
        // Verificar si el usuario está logueado
        if (!Session::has('user_id')) {
            return response()->json([
                'success' => false,
                'message' => 'Debes iniciar sesión para agregar productos al carrito'
            ]);
        }

        $userId = Session::get('user_id');
        $productId = $request->product_id;
        $quantity = $request->quantity ?? 1;

        // Verificar si el producto existe
        $product = Product::find($productId);
        if (!$product) {
            return response()->json([
                'success' => false,
                'message' => 'Producto no encontrado'
            ]);
        }

        // Verificar si el producto ya está en el carrito
        $cartItem = Cart::where('user_id', $userId)
                       ->where('product_id', $productId)
                       ->first();

        if ($cartItem) {
            // Actualizar cantidad
            $cartItem->quantity += $quantity;
            $cartItem->save();
            $message = 'Cantidad actualizada en el carrito';
        } else {
            // Crear nuevo item en carrito
            Cart::create([
                'user_id' => $userId,
                'product_id' => $productId,
                'quantity' => $quantity
            ]);
            $message = 'Producto agregado al carrito';
        }

        // Obtener cantidad total de items en carrito
        $cartCount = Cart::where('user_id', $userId)->sum('quantity');

        return response()->json([
            'success' => true,
            'message' => $message,
            'cart_count' => $cartCount
        ]);
    }

    public function getCartCount()
    {
        if (!Session::has('user_id')) {
            return response()->json(['count' => 0]);
        }

        $userId = Session::get('user_id');
        $count = Cart::where('user_id', $userId)->sum('quantity');

        return response()->json(['count' => $count]);
    }

    public function getCartItems()
    {
        if (!Session::has('user_id')) {
            return response()->json(['success' => false, 'message' => 'No autorizado']);
        }

        $userId = Session::get('user_id');
        $cartItems = Cart::with('product')->where('user_id', $userId)->get();
        
        $items = $cartItems->map(function($item) {
            return [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'name' => $item->product->name,
                'price' => $item->product->price,
                'quantity' => $item->quantity,
                'subtotal' => $item->product->price * $item->quantity,
                'image' => $item->product->image
            ];
        });

        $total = $items->sum('subtotal');

        return response()->json([
            'success' => true,
            'items' => $items,
            'total' => $total,
            'formatted_total' => 'S/ ' . number_format($total, 2)
        ]);
    }

    public function updateCartItem(Request $request)
    {
        if (!Session::has('user_id')) {
            return response()->json(['success' => false, 'message' => 'No autorizado']);
        }

        $userId = Session::get('user_id');
        $cartItem = Cart::where('user_id', $userId)->where('id', $request->cart_item_id)->first();

        if (!$cartItem) {
            return response()->json(['success' => false, 'message' => 'Item no encontrado']);
        }

        if ($request->quantity <= 0) {
            $cartItem->delete();
            $message = 'Producto eliminado del carrito';
        } else {
            $cartItem->quantity = $request->quantity;
            $cartItem->save();
            $message = 'Cantidad actualizada';
        }

        // Obtener nuevo total del carrito
        $cartCount = Cart::where('user_id', $userId)->sum('quantity');

        return response()->json([
            'success' => true,
            'message' => $message,
            'cart_count' => $cartCount
        ]);
    }

    public function removeCartItem(Request $request)
    {
        if (!Session::has('user_id')) {
            return response()->json(['success' => false, 'message' => 'No autorizado']);
        }

        $userId = Session::get('user_id');
        $cartItem = Cart::where('user_id', $userId)->where('id', $request->cart_item_id)->first();

        if (!$cartItem) {
            return response()->json(['success' => false, 'message' => 'Item no encontrado']);
        }

        $cartItem->delete();
        $cartCount = Cart::where('user_id', $userId)->sum('quantity');

        return response()->json([
            'success' => true,
            'message' => 'Producto eliminado del carrito',
            'cart_count' => $cartCount
        ]);
    }

    public function checkout()
    {
        if (!Session::has('user_id')) {
            return redirect()->route('login');
        }

        // Verificar si hay items en el carrito
        $userId = Session::get('user_id');
        $cartItems = Cart::where('user_id', $userId)->get();
        
        if ($cartItems->isEmpty()) {
            // Limpiar variables de sesión si el carrito está vacío
            Session::forget(['processing_order', 'order_completed', 'last_order_id']);
            return redirect()->route('Producto.index')->with('error', 'Tu carrito está vacío');
        }

        // Verificar si ya se completó una orden recientemente (para evitar refrescos)
        if (Session::has('order_completed')) {
            $lastOrderId = Session::get('last_order_id');
            return redirect()->route('order.success', ['order' => $lastOrderId]);
        }

        // Si hay items en el carrito, permitir el checkout (limpiar processing_order si existe)
        // Esto permite al usuario reintentar el checkout si salió anteriormente
        Session::forget('processing_order');

        return view('checkout.index');
    }

    public function processOrder(Request $request)
    {
        if (!Session::has('user_id')) {
            return response()->json(['success' => false, 'message' => 'No autorizado']);
        }

        try {
            $userId = Session::get('user_id');
            
            // Validar que hay items en el carrito
            $cartItems = Cart::with('product')->where('user_id', $userId)->get();
            
            if ($cartItems->isEmpty()) {
                return response()->json(['success' => false, 'message' => 'El carrito está vacío']);
            }

            // Calcular totales
            $subtotal = $cartItems->sum(function($item) {
                return $item->product->price * $item->quantity;
            });
            
            $shippingCost = 0.00; // Envío gratis
            $total = $subtotal + $shippingCost;

            DB::beginTransaction();

            // Crear la orden
            $order = Order::create([
                'user_id' => $userId,
                'order_number' => Order::generateOrderNumber(),
                'full_name' => $request->full_name,
                'phone' => $request->phone,
                'address' => $request->address,
                'district' => $request->district,
                'payment_method' => $request->payment_method,
                'subtotal' => $subtotal,
                'shipping_cost' => $shippingCost,
                'total' => $total,
                'status' => 'pending'
            ]);

            // Crear los items de la orden
            foreach ($cartItems as $cartItem) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $cartItem->product_id,
                    'product_name' => $cartItem->product->name,
                    'product_price' => $cartItem->product->price,
                    'quantity' => $cartItem->quantity,
                    'subtotal' => $cartItem->product->price * $cartItem->quantity
                ]);
            }

            // Crear el registro de pago
            $paymentData = [
                'order_id' => $order->id,
                'payment_method' => $request->payment_method,
                'amount' => $total,
                'payment_status' => 'completed'
            ];

            // Si es pago con tarjeta, agregar información adicional
            if ($request->payment_method === 'tarjeta' && $request->has('card_number')) {
                $paymentData['card_number'] = $request->card_number;
                $paymentData['card_holder'] = $request->card_holder;
                $paymentData['card_type'] = $request->card_type;
                $paymentData['transaction_id'] = Payment::generateTransactionId();
                
                // Simular validación de tarjeta (en producción aquí iría la integración con procesador de pagos)
                $paymentData['payment_status'] = 'completed';
            }

            Payment::create($paymentData);

            // Limpiar el carrito
            Cart::where('user_id', $userId)->delete();

            // Marcar que se completó una orden para prevenir regresos
            Session::put('order_completed', true);
            Session::put('last_order_id', $order->order_number);
            
            // Limpiar datos temporales de checkout
            Session::forget('processing_order');

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Orden procesada exitosamente',
                'order_id' => $order->order_number,
                'redirect_url' => route('order.success', ['order' => $order->order_number])
            ]);
               

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Error al procesar la orden: ' . $e->getMessage()
            ]);
        }
    }

    public function orderSuccess($orderNumber)
    {
        if (!Session::has('user_id')) {
            return redirect()->route('login');
        }

        // Verificar que la orden pertenece al usuario logueado
        $userId = Session::get('user_id');
        $order = Order::where('order_number', $orderNumber)
                     ->where('user_id', $userId)
                     ->first();

        if (!$order) {
            return redirect()->route('index')->with('error', 'Orden no encontrada');
        }

        // Verificar que realmente se completó una orden recientemente
        if (!Session::has('order_completed') || Session::get('last_order_id') !== $orderNumber) {
            return redirect()->route('index')->with('info', 'Esta orden ya fue procesada anteriormente');
        }

        return view('order.success', compact('order'));
    }

    public function clearOrderSession()
    {
        Session::forget(['order_completed', 'last_order_id', 'processing_order']);
        return response()->json(['success' => true]);
    }

    public function getCategories()
    {
        $categories = Category::withCount('products')->get();
        
        return response()->json([
            'success' => true,
            'categories' => $categories
        ]);
    }
}