<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Models\Order;

class UserController extends Controller
{
    public function profile()
    {
        if (!Session::has('user_id')) {
            return redirect()->route('login');
        }

        $userId = Session::get('user_id');
        $user = User::find($userId);
        
        if (!$user) {
            Session::flush();
            return redirect()->route('login');
        }

        return view('user.profile', compact('user'));
    }

    public function orders()
    {
        if (!Session::has('user_id')) {
            return redirect()->route('login');
        }

        $userId = Session::get('user_id');
        $orders = Order::with('orderItems.product')
                      ->where('user_id', $userId)
                      ->orderBy('created_at', 'desc')
                      ->paginate(10);

        return view('user.orders', compact('orders'));
    }

    public function settings()
    {
        if (!Session::has('user_id')) {
            return redirect()->route('login');
        }

        $userId = Session::get('user_id');
        $user = User::find($userId);

        return view('user.settings', compact('user'));
    }

    public function updateProfile(Request $request)
    {
        if (!Session::has('user_id')) {
            return response()->json(['success' => false, 'message' => 'No autorizado']);
        }

        $userId = Session::get('user_id');
        $user = User::find($userId);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $userId,
            'phone' => 'nullable|string|max:20'
        ]);

        try {
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone
            ]);

            // Actualizar sesión
            Session::put('user_name', $user->name);
            Session::put('user_email', $user->email);

            return response()->json([
                'success' => true,
                'message' => 'Perfil actualizado exitosamente'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar perfil: ' . $e->getMessage()
            ]);
        }
    }

    public function updatePassword(Request $request)
    {
        if (!Session::has('user_id')) {
            return response()->json(['success' => false, 'message' => 'No autorizado']);
        }

        $userId = Session::get('user_id');
        $user = User::find($userId);

        $request->validate([
            'current_password' => 'required',
            'new_password' => 'required|string|min:6',
            'new_password_confirmation' => 'required|same:new_password'
        ]);

        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'La contraseña actual es incorrecta'
            ]);
        }

        try {
            $user->update([
                'password' => Hash::make($request->new_password)
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Contraseña actualizada exitosamente'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar contraseña: ' . $e->getMessage()
            ]);
        }
    }
}
