<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class BlogPost extends Model
{
    use HasFactory;

    protected $table = 'blog_posts';

    protected $fillable = [
        'title',
        'slug',
        'excerpt',
        'content',
        'image_url',
        'category',
        'author',
        'status',
        'featured',
        'meta_title',
        'meta_description',
        'views_count',
        'published_at'
    ];

    protected $casts = [
        'featured' => 'boolean',
        'views_count' => 'integer',
        'published_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // Scopes para filtrar posts
    public function scopePublished($query)
    {
        return $query->where('status', 'publicado')
                    ->where('published_at', '<=', now());
    }

    public function scopeFeatured($query)
    {
        return $query->where('featured', true);
    }

    public function scopeByCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    public function scopeRecent($query, $limit = 5)
    {
        return $query->orderBy('published_at', 'desc')->limit($limit);
    }

    // Generar slug automáticamente
    public function setTitleAttribute($value)
    {
        $this->attributes['title'] = $value;
        if (!$this->slug) {
            $this->attributes['slug'] = \Str::slug($value);
        }
    }

    // Incrementar contador de vistas
    public function incrementViews()
    {
        $this->increment('views_count');
    }

    // Obtener fecha formateada
    public function getFormattedDateAttribute()
    {
        return $this->published_at ? $this->published_at->format('d M Y') : $this->created_at->format('d M Y');
    }

    // Obtener fecha en español
    public function getSpanishDateAttribute()
    {
        $months = [
            1 => 'Enero', 2 => 'Febrero', 3 => 'Marzo', 4 => 'Abril',
            5 => 'Mayo', 6 => 'Junio', 7 => 'Julio', 8 => 'Agosto',
            9 => 'Septiembre', 10 => 'Octubre', 11 => 'Noviembre', 12 => 'Diciembre'
        ];

        $date = $this->published_at ?: $this->created_at;
        $day = $date->day;
        $month = $months[$date->month];
        $year = $date->year;

        return "{$day} {$month} {$year}";
    }

    // Obtener URL del post
    public function getUrlAttribute()
    {
        return route('blog.show', $this->slug);
    }

    // Obtener extracto limitado
    public function getShortExcerptAttribute($limit = 150)
    {
        return \Str::limit($this->excerpt, $limit);
    }

    // Obtener tiempo de lectura estimado
    public function getReadTimeAttribute()
    {
        $words = str_word_count(strip_tags($this->content));
        $minutes = ceil($words / 200); // Promedio 200 palabras por minuto
        return $minutes . ' min de lectura';
    }

    // Verificar si el post está publicado
    public function isPublished()
    {
        return $this->status === 'publicado' && 
               ($this->published_at === null || $this->published_at <= now());
    }

    // Obtener posts relacionados por categoría
    public function getRelatedPosts($limit = 3)
    {
        return self::published()
                   ->where('category', $this->category)
                   ->where('id', '!=', $this->id)
                   ->recent($limit)
                   ->get();
    }

    // Obtener todos los posts para el home (destacados primero)
    public static function getHomePagePosts($limit = 3)
    {
        return self::published()
                   ->orderByDesc('featured')
                   ->orderByDesc('published_at')
                   ->limit($limit)
                   ->get();
    }

    // Obtener categorías disponibles
    public static function getCategories()
    {
        return self::published()
                   ->distinct()
                   ->pluck('category')
                   ->toArray();
    }
}
