<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    use HasFactory;
    
    protected $table = 'orders';
    
    protected $fillable = [
        'user_id',
        'order_number',
        'full_name',
        'phone',
        'address',
        'district',
        'payment_method',
        'subtotal',
        'shipping_cost',
        'total',
        'status'
    ];
    
    protected $casts = [
        'subtotal' => 'decimal:2',
        'shipping_cost' => 'decimal:2',
        'total' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];
    
    // Estados válidos (valores exactos de la base de datos)
    public const STATUS_PENDING = 'pending';
    public const STATUS_CONFIRMED = 'confirmed';
    public const STATUS_PROCESSING = 'processing';
    public const STATUS_SHIPPED = 'shipped';
    public const STATUS_DELIVERED = 'delivered';
    public const STATUS_CANCELLED = 'cancelled';
    
    public static function getValidStatuses()
    {
        return [
            self::STATUS_PENDING,
            self::STATUS_CONFIRMED,
            self::STATUS_PROCESSING,
            self::STATUS_SHIPPED,
            self::STATUS_DELIVERED,
            self::STATUS_CANCELLED
        ];
    }
    
    public static function getStatusLabels()
    {
        return [
            self::STATUS_PENDING => 'pending',
            self::STATUS_CONFIRMED => 'confirmed',
            self::STATUS_PROCESSING => 'processing',
            self::STATUS_SHIPPED => 'shipped',
            self::STATUS_DELIVERED => 'delivered',
            self::STATUS_CANCELLED => 'cancelled'
        ];
    }
    
    public function getStatusLabelAttribute()
    {
        $labels = self::getStatusLabels();
        return $labels[$this->status] ?? $this->status;
    }
    
    public function getStatusClassAttribute()
    {
        $classes = [
            self::STATUS_PENDING => 'badge-warning',
            self::STATUS_CONFIRMED => 'badge-info',
            self::STATUS_PROCESSING => 'badge-primary',
            self::STATUS_SHIPPED => 'badge-secondary',
            self::STATUS_DELIVERED => 'badge-success',
            self::STATUS_CANCELLED => 'badge-danger'
        ];
        return $classes[$this->status] ?? 'badge-light';
    }
    
    // Relación con el usuario
    public function user()
    {
        return $this->belongsTo(User::class);
    }
    
    // Relación con los items de la orden
    public function orderItems()
    {
        return $this->hasMany(OrderItem::class);
    }
    
    // Generar número de orden único
    public static function generateOrderNumber()
    {
        $prefix = 'EM';
        $timestamp = now()->format('ymd');
        $random = str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        return $prefix . $timestamp . $random;
    }
}
