<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Quote extends Model
{
    use HasFactory;

    protected $table = 'quotes';

    protected $fillable = [
        'name',
        'email',
        'phone',
        'product_name',
        'product_category',
        'quantity',
        'budget_range',
        'message',
        'status',
        'priority'
    ];

    protected $casts = [
        'quantity' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // Scopes para filtrar por estado
    public function scopePendientes($query)
    {
        return $query->where('status', 'pendiente');
    }

    public function scopeProcesando($query)
    {
        return $query->where('status', 'procesando');
    }

    public function scopeEnviados($query)
    {
        return $query->where('status', 'enviado');
    }

    // Scopes para filtrar por prioridad
    public function scopeAltaPrioridad($query)
    {
        return $query->where('priority', 'alta');
    }

    public function scopeMediaPrioridad($query)
    {
        return $query->where('priority', 'media');
    }

    // Cambiar estado de la cotización
    public function markAsProcessing()
    {
        $this->update(['status' => 'procesando']);
    }

    public function markAsSent()
    {
        $this->update(['status' => 'enviado']);
    }

    public function markAsClosed()
    {
        $this->update(['status' => 'cerrado']);
    }

    // Obtener nombre del estado en español
    public function getStatusNameAttribute()
    {
        $statuses = [
            'pendiente' => 'Pendiente',
            'procesando' => 'Procesando',
            'enviado' => 'Enviado',
            'cerrado' => 'Cerrado'
        ];

        return $statuses[$this->status] ?? $this->status;
    }

    // Obtener clase CSS para el estado
    public function getStatusClassAttribute()
    {
        $classes = [
            'pendiente' => 'badge-warning',
            'procesando' => 'badge-info',
            'enviado' => 'badge-success',
            'cerrado' => 'badge-secondary'
        ];

        return $classes[$this->status] ?? 'badge-light';
    }

    // Obtener clase CSS para la prioridad
    public function getPriorityClassAttribute()
    {
        $classes = [
            'baja' => 'badge-success',
            'media' => 'badge-warning',
            'alta' => 'badge-danger'
        ];

        return $classes[$this->priority] ?? 'badge-light';
    }
}
