<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;

class User extends Authenticatable
{
    use HasFactory, Notifiable;

    protected $table = 'users';

    protected $fillable = [
        'name',
        'email',
        'password',
        'phone',
        'rol',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // Mutador para hashear la contraseña automáticamente
    public function setPasswordAttribute($value)
    {
        $this->attributes['password'] = Hash::make($value);
    }

    // Verificar contraseña
    public function checkPassword($password)
    {
        return Hash::check($password, $this->password);
    }

    // Obtener el nombre completo
    public function getFullNameAttribute()
    {
        return $this->name;
    }

    // Verificar si el usuario está activo
    public function isActive()
    {
        return true; // Por ahora todos los usuarios están activos
    }

    // Verificar si el usuario es administrador
    public function isAdmin()
    {
        return $this->rol === 'administrador';
    }

    // Scope para búsqueda por email
    public function scopeByEmail($query, $email)
    {
        return $query->where('email', $email);
    }

    // Scope para administradores
    public function scopeAdmins($query)
    {
        return $query->where('rol', 'administrador');
    }

    // Scope para usuarios regulares
    public function scopeUsers($query)
    {
        return $query->where('rol', '!=', 'administrador');
    }

    // Relación con órdenes
    public function orders()
    {
        return $this->hasMany(\App\Models\Order::class);
    }

    // Relación con carrito
    public function cartItems()
    {
        return $this->hasMany(\App\Models\Cart::class);
    }
}
