// Cart functionality
let cartItems = 0;
let cartProducts = [];

// DOM Elements - with null checks
const cartCountElement = document.querySelector('.cart-count');
const cartBtn = document.querySelector('.cart-btn');
const addToCartButtons = document.querySelectorAll('.product-btn');
const modal = document.getElementById('products-modal');
const modalClose = document.querySelector('.modal-close');
const productsLink = document.querySelector('a[href="#productos"]');
const navProductsLink = document.querySelector('.nav-link[href="#productos"]');

// Initialize cart
function updateCartCount() {
    if (cartCountElement) {
        cartCountElement.textContent = cartItems;
        
        // Add animation
        cartCountElement.style.transform = 'scale(1.2)';
        setTimeout(() => {
            cartCountElement.style.transform = 'scale(1)';
        }, 200);
    }
}

// Add to cart functionality
function addToCart(productName, price) {
    cartItems++;
    cartProducts.push({ name: productName, price: price });
    updateCartCount();
    
    // Show success message
    showNotification(`${productName} agregado al carrito`, 'success');
}

// Show notification
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#10B981' : '#4266E8'};
        color: white;
        padding: 12px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        z-index: 3000;
        font-weight: 500;
        transform: translateX(100%);
        transition: all 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    // Animate in
    setTimeout(() => {
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    // Remove after 3 seconds
    setTimeout(() => {
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// Event listeners for add to cart buttons
addToCartButtons.forEach((button, index) => {
    button.addEventListener('click', () => {
        const productCard = button.closest('.product-card');
        const productName = productCard.querySelector('.product-name').textContent;
        const productPrice = productCard.querySelector('.product-price').textContent;
        addToCart(productName, productPrice);
    });
});

// Products modal functionality
function openProductsModal() {
    modal.style.display = 'block';
    document.body.style.overflow = 'hidden';
}

function closeProductsModal() {
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
}

// Event listeners for products modal
if (productsLink) {
    productsLink.addEventListener('click', (e) => {
        e.preventDefault();
        openProductsModal();
    });
}

if (navProductsLink) {
    navProductsLink.addEventListener('click', (e) => {
        e.preventDefault();
        openProductsModal();
    });
}

// Add null checks for modal elements
if (modalClose) {
    modalClose.addEventListener('click', closeProductsModal);
}

if (modal) {
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            closeProductsModal();
        }
    });
}

// Close modal with Escape key
document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape' && modal && modal.style.display === 'block') {
        closeProductsModal();
    }
});

// Newsletter form - with null check
const newsletterForm = document.querySelector('.newsletter-form');
if (newsletterForm) {
    newsletterForm.addEventListener('submit', (e) => {
        e.preventDefault();
        const email = e.target.querySelector('input').value;
        if (email) {
            showNotification('¡Gracias por suscribirte! Recibirás nuestras ofertas especiales.', 'success');
            e.target.querySelector('input').value = '';
        }
    });
}

// Smooth scroll for navigation links
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        e.preventDefault();
        const targetId = this.getAttribute('href');
        if (targetId !== '#productos') { // Don't smooth scroll for products modal
            const targetElement = document.querySelector(targetId);
            if (targetElement) {
                const offsetTop = targetElement.offsetTop - 100; // Account for fixed header
                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        }
    });
});

// Scroll animations
const observerOptions = {
    threshold: 0.1,
    rootMargin: '0px 0px -50px 0px'
};

const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting) {
            entry.target.classList.add('visible');
        }
    });
}, observerOptions);

// Add fade-in class to elements and observe them
window.addEventListener('load', () => {
    const animateElements = document.querySelectorAll('.category-card, .product-card, .contact-card');
    animateElements.forEach(el => {
        el.classList.add('fade-in');
        observer.observe(el);
    });
});

// Category filter functionality (for products page)
const categoryLinks = document.querySelectorAll('.category-link');
categoryLinks.forEach(link => {
    link.addEventListener('click', (e) => {
        e.preventDefault();
        
        // Update active state
        categoryLinks.forEach(l => l.classList.remove('active'));
        link.classList.add('active');
        
        // Filter products (simulation)
        const category = link.textContent.trim();
        showNotification(`Filtrando por: ${category}`, 'info');
    });
});

// Header scroll effect
let lastScrollTop = 0;
const header = document.querySelector('.header');

window.addEventListener('scroll', () => {
    const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
    
    if (scrollTop > 100) {
        header.style.background = 'rgba(66, 102, 232, 0.95)';
        header.style.backdropFilter = 'blur(10px)';
    } else {
        header.style.background = '#4266E8';
        header.style.backdropFilter = 'none';
    }
    
    lastScrollTop = scrollTop;
});

// Product card hover effects
const productCards = document.querySelectorAll('.product-card, .product-card-detailed');
productCards.forEach(card => {
    card.addEventListener('mouseenter', () => {
        card.style.transform = 'translateY(-8px) scale(1.02)';
    });
    
    card.addEventListener('mouseleave', () => {
        card.style.transform = 'translateY(0) scale(1)';
    });
});

// Search functionality (placeholder)
const searchBtn = document.querySelector('.search-btn');
searchBtn.addEventListener('click', () => {
    showNotification('Función de búsqueda próximamente', 'info');
});

// User account functionality (placeholder)
const userBtn = document.querySelector('.user-btn');
userBtn.addEventListener('click', () => {
    showNotification('Iniciar sesión próximamente', 'info');
});

// Mobile menu toggle (for future enhancement)
function toggleMobileMenu() {
    const nav = document.querySelector('.main-nav');
    nav.classList.toggle('mobile-active');
}

// Add loading effect to buttons
const buttons = document.querySelectorAll('.btn');
buttons.forEach(button => {
    button.addEventListener('click', function(e) {
        if (!this.classList.contains('loading')) {
            this.classList.add('loading');
            
            // Create ripple effect
            const ripple = document.createElement('span');
            ripple.className = 'ripple';
            
            const rect = this.getBoundingClientRect();
            const size = Math.max(rect.width, rect.height);
            const x = e.clientX - rect.left - size / 2;
            const y = e.clientY - rect.top - size / 2;
            
            ripple.style.cssText = `
                position: absolute;
                top: ${y}px;
                left: ${x}px;
                width: ${size}px;
                height: ${size}px;
                border-radius: 50%;
                background: rgba(255,255,255,0.3);
                transform: scale(0);
                animation: ripple 0.6s ease-out;
                pointer-events: none;
            `;
            
            this.style.position = 'relative';
            this.style.overflow = 'hidden';
            this.appendChild(ripple);
            
            setTimeout(() => {
                this.classList.remove('loading');
                if (ripple.parentNode) {
                    ripple.parentNode.removeChild(ripple);
                }
            }, 600);
        }
    });
});

// Add ripple animation
const style = document.createElement('style');
style.textContent = `
    @keyframes ripple {
        to {
            transform: scale(2);
            opacity: 0;
        }
    }
    
    .loading {
        pointer-events: none;
    }
`;
document.head.appendChild(style);

// Filter Accordion Functionality
document.addEventListener('DOMContentLoaded', function() {
    // Mobile filter toggle
    const toggleFiltersBtn = document.getElementById('toggleFilters');
    const filtersSidebar = document.getElementById('filtersSidebar');
    const toggleIcon = toggleFiltersBtn?.querySelector('.toggle-icon');

    if (toggleFiltersBtn && filtersSidebar) {
        toggleFiltersBtn.addEventListener('click', function() {
            filtersSidebar.classList.toggle('show');
            toggleIcon?.classList.toggle('rotated');
        });
    }

    // Filter accordion toggles
    const filterToggles = document.querySelectorAll('.filter-toggle');
    
    filterToggles.forEach(toggle => {
        toggle.addEventListener('click', function() {
            const targetId = this.getAttribute('data-target');
            const targetContent = document.getElementById(targetId);
            const chevronIcon = this.querySelector('i');
            
            if (targetContent) {
                // Close other accordions (optional - for single accordion behavior)
                // Uncomment the following lines if you want only one accordion open at a time
                /*
                filterToggles.forEach(otherToggle => {
                    if (otherToggle !== this) {
                        const otherTargetId = otherToggle.getAttribute('data-target');
                        const otherContent = document.getElementById(otherTargetId);
                        const otherIcon = otherToggle.querySelector('i');
                        
                        if (otherContent && otherContent.classList.contains('show')) {
                            otherContent.classList.remove('show');
                            otherToggle.classList.remove('active');
                            if (otherIcon) otherIcon.style.transform = 'rotate(0deg)';
                        }
                    }
                });
                */
                
                // Toggle current accordion
                targetContent.classList.toggle('show');
                this.classList.toggle('active');
                
                if (chevronIcon) {
                    chevronIcon.style.transform = targetContent.classList.contains('show') 
                        ? 'rotate(180deg)' 
                        : 'rotate(0deg)';
                }
            }
        });
    });

    // Initialize first accordion item as open on mobile
    const firstFilterToggle = document.querySelector('.filter-toggle');
    if (firstFilterToggle && window.innerWidth <= 991) {
        // Don't auto-open on mobile to save space
        // firstFilterToggle.click();
    }

    // Handle window resize
    let resizeTimeout;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(function() {
            if (window.innerWidth > 991) {
                // Desktop view - ensure sidebar is visible and accordions are reset
                if (filtersSidebar) {
                    filtersSidebar.classList.remove('show');
                }
                
                // Reset accordion states on desktop
                const filterContents = document.querySelectorAll('.filter-content');
                const filterTogglesAll = document.querySelectorAll('.filter-toggle');
                
                filterContents.forEach(content => {
                    content.classList.remove('show');
                });
                
                filterTogglesAll.forEach(toggle => {
                    toggle.classList.remove('active');
                    const icon = toggle.querySelector('i');
                    if (icon) icon.style.transform = 'rotate(0deg)';
                });
                
                if (toggleIcon) {
                    toggleIcon.classList.remove('rotated');
                }
            }
        }, 250);
    });
});

console.log('ElectroMart website loaded successfully!');