@extends('administrador.include.template')

@section('title', 'Información de la Empresa')

@section('breadcrumb')
    <span class="breadcrumb-item">Configuración</span>
    <span class="breadcrumb-item active">Información de la Empresa</span>
@endsection

@push('styles')
<style>
    .company-info-container {
        max-width: 800px;
        margin: 0 auto;
    }
    
    .company-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 2rem;
    }
    
    .company-title {
        font-size: 1.5rem;
        font-weight: 600;
        color: var(--gray-900);
    }
    
    .company-form {
        background: white;
        border-radius: 12px;
        padding: 2rem;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    }
    
    .form-section {
        margin-bottom: 2rem;
    }
    
    .section-title {
        font-size: 1.1rem;
        font-weight: 600;
        color: var(--gray-800);
        margin-bottom: 1rem;
        padding-bottom: 0.5rem;
        border-bottom: 2px solid #667eea;
    }
    
    .form-row {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1rem;
        margin-bottom: 1rem;
    }
    
    .form-group {
        display: flex;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .form-group.full-width {
        grid-column: 1 / -1;
    }
    
    .form-label {
        font-size: 0.875rem;
        font-weight: 600;
        color: var(--gray-700);
    }
    
    .form-label.required::after {
        content: ' *';
        color: #e53e3e;
    }
    
    .form-input {
        padding: 0.75rem;
        border: 2px solid var(--gray-200);
        border-radius: 6px;
        font-size: 0.875rem;
        transition: all 0.2s;
        background: white;
    }
    
    .form-input:focus {
        outline: none;
        border-color: #667eea;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }
    
    .form-input.error {
        border-color: #e53e3e;
    }
    
    .form-textarea {
        min-height: 100px;
        resize: vertical;
        font-family: inherit;
    }
    
    .form-checkbox-group {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.75rem;
        background: var(--gray-50);
        border-radius: 6px;
        border: 2px solid var(--gray-200);
        transition: all 0.2s;
    }
    
    .form-checkbox-group:hover {
        background: var(--gray-100);
    }
    
    .form-checkbox {
        width: 1.25rem;
        height: 1.25rem;
        accent-color: #667eea;
    }
    
    .checkbox-label {
        font-size: 0.875rem;
        color: var(--gray-700);
        cursor: pointer;
    }
    
    .error-message {
        color: #e53e3e;
        font-size: 0.75rem;
        margin-top: 0.25rem;
    }
    
    .form-actions {
        display: flex;
        justify-content: end;
        gap: 1rem;
        margin-top: 2rem;
        padding-top: 2rem;
        border-top: 1px solid var(--gray-200);
    }
    
    .btn {
        padding: 0.75rem 1.5rem;
        border-radius: 6px;
        font-size: 0.875rem;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s;
        border: none;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }
    
    .btn-primary {
        background: #667eea;
        color: white;
    }
    
    .btn-primary:hover {
        background: #5a67d8;
        transform: translateY(-1px);
    }
    
    .btn-primary:disabled {
        background: #a0aec0;
        cursor: not-allowed;
        transform: none;
    }
    
    .btn-secondary {
        background: var(--gray-100);
        color: var(--gray-700);
        border: 2px solid var(--gray-200);
    }
    
    .btn-secondary:hover {
        background: var(--gray-200);
        transform: translateY(-1px);
    }
    
    .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.5);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 1000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s;
    }
    
    .loading-overlay.active {
        opacity: 1;
        visibility: visible;
    }
    
    .loading-spinner {
        background: white;
        padding: 2rem;
        border-radius: 12px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 1rem;
    }
    
    .spinner {
        width: 40px;
        height: 40px;
        border: 4px solid #f3f3f3;
        border-top: 4px solid #667eea;
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
    
    .info-card {
        background: #f0f9ff;
        border: 1px solid #0ea5e9;
        border-radius: 8px;
        padding: 1rem;
        margin-bottom: 2rem;
    }
    
    .info-card-title {
        font-weight: 600;
        color: #0c4a6e;
        margin-bottom: 0.5rem;
    }
    
    .info-card-text {
        color: #0369a1;
        font-size: 0.875rem;
    }
    
    @media (max-width: 768px) {
        .company-info-container {
            padding: 0 1rem;
        }
        
        .company-form {
            padding: 1.5rem;
        }
        
        .form-row {
            grid-template-columns: 1fr;
        }
        
        .form-actions {
            flex-direction: column;
        }
        
        .company-header {
            flex-direction: column;
            gap: 1rem;
            align-items: stretch;
        }
    }
</style>
@endpush

@section('content')
<div class="company-info-container">
    <!-- Header -->
    <div class="company-header">
        <h1 class="company-title">
            <i class="fas fa-building mr-2" style="color: #667eea;"></i>
            Información de la Empresa
        </h1>
    </div>
    
    <!-- Info Card -->
    <div class="info-card">
        <div class="info-card-title">
            <i class="fas fa-info-circle mr-1"></i>
            Configuración de Empresa
        </div>
        <div class="info-card-text">
            Aquí puedes actualizar la información principal de tu empresa que se mostrará en el sitio web y en los documentos generados.
        </div>
    </div>
    
    <!-- Formulario -->
    <div class="company-form">
        <form id="companyForm">
            <!-- Información Básica -->
            <div class="form-section">
                <h3 class="section-title">
                    <i class="fas fa-info-circle mr-2"></i>
                    Información Básica
                </h3>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="name" class="form-label required">Nombre de la Empresa</label>
                        <input type="text" id="name" name="name" class="form-input" placeholder="Electrodomésticos La Casa">
                        <div class="error-message" id="name-error"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="email" class="form-label required">Email Corporativo</label>
                        <input type="email" id="email" name="email" class="form-input" placeholder="info@empresa.com">
                        <div class="error-message" id="email-error"></div>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="phone" class="form-label required">Teléfono Principal</label>
                        <input type="tel" id="phone" name="phone" class="form-input" placeholder="+1 234 567 8900">
                        <div class="error-message" id="phone-error"></div>
                    </div>
                </div>
            </div>
            
            <!-- Información de Contacto -->
            <div class="form-section">
                <h3 class="section-title">
                    <i class="fas fa-map-marker-alt mr-2"></i>
                    Ubicación y Horarios
                </h3>
                
                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="address" class="form-label required">Dirección Completa</label>
                        <input type="text" id="address" name="address" class="form-input" placeholder="Calle Principal #123, Ciudad, Estado, Código Postal">
                        <div class="error-message" id="address-error"></div>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="hours" class="form-label required">Horarios de Atención</label>
                        <textarea id="hours" name="hours" class="form-input form-textarea" placeholder="Lunes a Viernes: 9:00 AM - 6:00 PM&#10;Sábados: 9:00 AM - 2:00 PM&#10;Domingos: Cerrado"></textarea>
                        <div class="error-message" id="hours-error"></div>
                    </div>
                </div>
            </div>
            
            <!-- Descripción -->
            <div class="form-section">
                <h3 class="section-title">
                    <i class="fas fa-file-alt mr-2"></i>
                    Descripción de la Empresa
                </h3>
                
                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="description" class="form-label">Descripción (Opcional)</label>
                        <textarea id="description" name="description" class="form-input form-textarea" placeholder="Breve descripción de tu empresa, servicios y valores..."></textarea>
                        <div class="error-message" id="description-error"></div>
                    </div>
                </div>
            </div>
            
            <!-- Estado -->
            <div class="form-section">
                <h3 class="section-title">
                    <i class="fas fa-toggle-on mr-2"></i>
                    Estado
                </h3>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-checkbox-group">
                            <input type="checkbox" id="is_active" name="is_active" class="form-checkbox" checked>
                            <span class="checkbox-label">Información activa (visible en el sitio web)</span>
                        </label>
                    </div>
                </div>
            </div>
            
            <!-- Acciones -->
            <div class="form-actions">
                <button type="button" class="btn btn-secondary" onclick="resetForm()">
                    <i class="fas fa-undo"></i>
                    Restablecer
                </button>
                <button type="submit" class="btn btn-primary" id="saveBtn">
                    <i class="fas fa-save"></i>
                    Guardar Cambios
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay" id="loadingOverlay">
    <div class="loading-spinner">
        <div class="spinner"></div>
        <div>Cargando información...</div>
    </div>
</div>
@endsection

@push('scripts')
<script>
let originalData = {};

// Cargar datos al iniciar
document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 Iniciando gestión de información de empresa...');
    loadCompanyInfo();
    
    // Event listener para el formulario
    document.getElementById('companyForm').addEventListener('submit', saveCompanyInfo);
});

// Cargar información de la empresa
async function loadCompanyInfo() {
    try {
        showLoading(true);
        console.log('📊 Cargando información de la empresa...');
        
        const response = await fetch('{{ route("admin.company-info.data") }}');
        const result = await response.json();
        
        if (result.success) {
            const data = result.data;
            originalData = { ...data };
            
            // Llenar el formulario
            document.getElementById('name').value = data.name || '';
            document.getElementById('email').value = data.email || '';
            document.getElementById('phone').value = data.phone || '';
            document.getElementById('address').value = data.address || '';
            document.getElementById('hours').value = data.hours || '';
            document.getElementById('description').value = data.description || '';
            document.getElementById('is_active').checked = data.is_active;
            
            console.log('✅ Información cargada');
        } else {
            throw new Error(result.message);
        }
    } catch (error) {
        console.error('❌ Error al cargar información:', error);
        showError('Error al cargar la información de la empresa');
    } finally {
        showLoading(false);
    }
}

// Guardar información de la empresa
async function saveCompanyInfo(event) {
    event.preventDefault();
    
    try {
        // Limpiar errores previos
        clearErrors();
        
        const saveBtn = document.getElementById('saveBtn');
        saveBtn.disabled = true;
        saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Guardando...';
        
        console.log('💾 Guardando información de la empresa...');
        
        const formData = new FormData(document.getElementById('companyForm'));
        
        const response = await fetch('{{ route("admin.company-info.update") }}', {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                name: formData.get('name'),
                email: formData.get('email'),
                phone: formData.get('phone'),
                address: formData.get('address'),
                hours: formData.get('hours'),
                description: formData.get('description'),
                is_active: formData.has('is_active')
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            originalData = { ...result.data };
            
            Swal.fire({
                title: '¡Actualizado!',
                text: result.message,
                icon: 'success',
                timer: 3000,
                showConfirmButton: false
            });
            
            console.log('✅ Información guardada exitosamente');
        } else {
            if (result.errors) {
                displayErrors(result.errors);
            } else {
                throw new Error(result.message);
            }
        }
    } catch (error) {
        console.error('❌ Error al guardar:', error);
        showError('Error al guardar la información');
    } finally {
        const saveBtn = document.getElementById('saveBtn');
        saveBtn.disabled = false;
        saveBtn.innerHTML = '<i class="fas fa-save"></i> Guardar Cambios';
    }
}

// Restablecer formulario
function resetForm() {
    Swal.fire({
        title: '¿Restablecer formulario?',
        text: 'Se perderán todos los cambios no guardados',
        icon: 'question',
        showCancelButton: true,
        confirmButtonText: 'Sí, restablecer',
        cancelButtonText: 'Cancelar',
        confirmButtonColor: '#667eea',
        cancelButtonColor: '#6c757d'
    }).then((result) => {
        if (result.isConfirmed) {
            // Restaurar valores originales
            document.getElementById('name').value = originalData.name || '';
            document.getElementById('email').value = originalData.email || '';
            document.getElementById('phone').value = originalData.phone || '';
            document.getElementById('address').value = originalData.address || '';
            document.getElementById('hours').value = originalData.hours || '';
            document.getElementById('description').value = originalData.description || '';
            document.getElementById('is_active').checked = originalData.is_active;
            
            clearErrors();
            console.log('🔄 Formulario restablecido');
        }
    });
}

// Mostrar/ocultar loading
function showLoading(show) {
    const overlay = document.getElementById('loadingOverlay');
    if (show) {
        overlay.classList.add('active');
    } else {
        overlay.classList.remove('active');
    }
}

// Mostrar errores de validación
function displayErrors(errors) {
    for (const [field, messages] of Object.entries(errors)) {
        const errorElement = document.getElementById(`${field}-error`);
        const inputElement = document.getElementById(field);
        
        if (errorElement && inputElement) {
            errorElement.textContent = messages[0];
            inputElement.classList.add('error');
        }
    }
}

// Limpiar errores
function clearErrors() {
    document.querySelectorAll('.error-message').forEach(el => el.textContent = '');
    document.querySelectorAll('.form-input').forEach(el => el.classList.remove('error'));
}

// Mostrar error general
function showError(message) {
    Swal.fire({
        title: 'Error',
        text: message,
        icon: 'error',
        confirmButtonColor: '#667eea'
    });
}
</script>
@endpush
