@extends('administrador.include.template')

@section('title', 'Gestión de Pedidos')

@section('breadcrumb')
    <span class="breadcrumb-item">Ventas</span>
    <span class="breadcrumb-item active">Pedidos</span>
@endsection

@push('styles')
<style>
    .orders-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 2rem;
    }
    
    .orders-title {
        font-size: 1.5rem;
        font-weight: 600;
        color: var(--gray-900);
    }
    
    .orders-table {
        background: white;
        border-radius: 12px;
        overflow: hidden;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    }
    
    .table-responsive {
        overflow-x: auto;
    }
    
    .table {
        width: 100%;
        margin-bottom: 0;
        color: var(--gray-700);
    }
    
    .table thead th {
        background: var(--gray-50);
        border-bottom: 1px solid var(--gray-200);
        padding: 1rem;
        font-weight: 600;
        text-transform: uppercase;
        font-size: 0.75rem;
        letter-spacing: 0.5px;
        color: var(--gray-600);
    }
    
    .table tbody td {
        padding: 1rem;
        border-bottom: 1px solid var(--gray-100);
        vertical-align: middle;
    }
    
    .table tbody tr:hover {
        background: var(--gray-50);
    }
    
    .badge {
        padding: 0.25rem 0.75rem;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 600;
    }
    
    .badge-warning {
        background: rgba(237, 137, 54, 0.1);
        color: #ed8936;
    }
    
    .badge-info {
        background: rgba(66, 153, 225, 0.1);
        color: #4299e1;
    }
    
    .badge-primary {
        background: rgba(102, 126, 234, 0.1);
        color: #667eea;
    }
    
    .badge-secondary {
        background: rgba(113, 128, 150, 0.1);
        color: #718096;
    }
    
    .badge-success {
        background: rgba(72, 187, 120, 0.1);
        color: #48bb78;
    }
    
    .badge-danger {
        background: rgba(245, 101, 101, 0.1);
        color: #f56565;
    }
    
    .action-buttons {
        display: flex;
        gap: 0.5rem;
    }
    
    .btn-action {
        width: 32px;
        height: 32px;
        border: none;
        border-radius: 6px;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 0.875rem;
        transition: all 0.3s ease;
    }
    
    .btn-view {
        background: rgba(66, 153, 225, 0.1);
        color: var(--info-color);
    }
    
    .btn-edit {
        background: rgba(72, 187, 120, 0.1);
        color: var(--success-color);
    }
    
    .btn-delete {
        background: rgba(245, 101, 101, 0.1);
        color: var(--danger-color);
    }
    
    .btn-action:hover {
        transform: scale(1.1);
    }
    
    .loading-spinner {
        display: flex;
        justify-content: center;
        align-items: center;
        height: 200px;
    }
    
    .spinner {
        width: 40px;
        height: 40px;
        border: 4px solid var(--gray-200);
        border-top: 4px solid var(--primary-color);
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
    
    .empty-state {
        text-align: center;
        padding: 3rem;
        color: var(--gray-500);
    }
    
    .empty-state i {
        font-size: 3rem;
        margin-bottom: 1rem;
        color: var(--gray-300);
    }
    
    .stats-container {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
        gap: 1rem;
        margin-bottom: 2rem;
    }
    
    .stat-card {
        background: white;
        padding: 1.5rem;
        border-radius: 8px;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        text-align: center;
    }
    
    .stat-number {
        font-size: 2rem;
        font-weight: 700;
        margin-bottom: 0.5rem;
    }
    
    .stat-label {
        font-size: 0.875rem;
        color: var(--gray-600);
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .order-number {
        font-family: 'Courier New', monospace;
        font-weight: bold;
        color: #667eea;
    }
    
    .customer-info {
        max-width: 200px;
    }
    
    .address-preview {
        max-width: 180px;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
        font-size: 0.875rem;
        color: var(--gray-600);
    }
</style>
@endpush

@section('content')
<div class="orders-container">
    <div class="orders-header">
        <h1 class="orders-title">
            <i class="fas fa-shopping-cart"></i>
            Gestión de Pedidos
        </h1>
    </div>
    
    <!-- Estadísticas -->
    <div class="stats-container" id="statsContainer">
        <div class="stat-card">
            <div class="stat-number" style="color: #667eea;" id="totalCount">0</div>
            <div class="stat-label">Total Pedidos</div>
        </div>
        <div class="stat-card">
            <div class="stat-number" style="color: #ed8936;" id="pendientesCount">0</div>
            <div class="stat-label">Pendientes</div>
        </div>
        <div class="stat-card">
            <div class="stat-number" style="color: #4299e1;" id="confirmadosCount">0</div>
            <div class="stat-label">Confirmados</div>
        </div>
        <div class="stat-card">
            <div class="stat-number" style="color: #48bb78;" id="entregadosCount">0</div>
            <div class="stat-label">Entregados</div>
        </div>
        <div class="stat-card">
            <div class="stat-number" style="color: #667eea;" id="ventasTotal">S/ 0.00</div>
            <div class="stat-label">Ventas Total</div>
        </div>
    </div>
    
    <div class="orders-table">
        <div class="table-responsive">
            <table class="table" id="ordersTable">
                <thead>
                    <tr>
                        <th>Pedido</th>
                        <th>Cliente</th>
                        <th>Contacto</th>
                        <th>Dirección</th>
                        <th>Pago</th>
                        <th>Total</th>
                        <th>Items</th>
                        <th>Estado</th>
                        <th>Fecha</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="ordersTableBody">
                    <tr>
                        <td colspan="10">
                            <div class="loading-spinner">
                                <div class="spinner"></div>
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Cambiar Estado -->
<div id="statusModal" style="display: none;">
    <form id="statusForm">
        <div style="margin-bottom: 1rem;">
            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Nuevo Estado *</label>
            <select id="orderStatus" name="status" 
                    style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                    required>
                <option value="pending">Pendiente</option>
                <option value="confirmed">Confirmado</option>
                <option value="processing">En Preparación</option>
                <option value="shipped">Enviado</option>
                <option value="delivered">Entregado</option>
                <option value="cancelled">Cancelado</option>
            </select>
        </div>
    </form>
</div>
@endsection

@push('scripts')
<script>
let currentOrderId = null;

// Cargar pedidos al cargar la página
document.addEventListener('DOMContentLoaded', function() {
    loadOrders();
    loadStats();
});

// Cargar estadísticas
async function loadStats() {
    try {
        const response = await fetch('{{ route("admin.orders.stats") }}');
        const result = await response.json();
        
        if (result.success) {
            const stats = result.data;
            document.getElementById('totalCount').textContent = stats.total;
            document.getElementById('pendientesCount').textContent = stats.pendientes;
            document.getElementById('confirmadosCount').textContent = stats.confirmados;
            document.getElementById('entregadosCount').textContent = stats.entregados;
            document.getElementById('ventasTotal').textContent = 'S/ ' + parseFloat(stats.ventas_total).toFixed(2);
        }
    } catch (error) {
        console.error('Error al cargar estadísticas:', error);
    }
}

// Cargar pedidos
async function loadOrders() {
    console.log('🔄 Iniciando carga de pedidos...');
    try {
        const url = '{{ route("admin.orders.data") }}';
        console.log('📡 URL de solicitud:', url);
        
        const response = await fetch(url);
        console.log('📊 Respuesta recibida:', response.status, response.statusText);
        
        const result = await response.json();
        console.log('📋 Datos recibidos:', result);
        
        const tbody = document.getElementById('ordersTableBody');
        
        if (result.success && result.data.length > 0) {
            console.log('✅ Pedidos encontrados:', result.data.length);
            
            tbody.innerHTML = result.data.map(order => `
                <tr>
                    <td><span class="order-number">${order.order_number}</span></td>
                    <td>
                        <div class="customer-info">
                            <strong>${order.full_name}</strong><br>
                            <small style="color: var(--gray-600);">${order.user_email}</small>
                        </div>
                    </td>
                    <td>${order.phone}</td>
                    <td>
                        <div class="address-preview" title="${order.address}, ${order.district}">
                            ${order.address}, ${order.district}
                        </div>
                    </td>
                    <td><span class="badge badge-info">${getPaymentMethodText(order.payment_method)}</span></td>
                    <td><strong>S/ ${order.total}</strong></td>
                    <td><span class="badge badge-secondary">${order.items_count} items</span></td>
                    <td><span class="badge ${order.status_class}">${getStatusText(order.status)}</span></td>
                    <td>${order.created_at}</td>
                    <td>
                        <div class="action-buttons">
                            <button class="btn-action btn-view" onclick="viewOrder(${order.id})" title="Ver detalles del pedido">
                                <i class="fas fa-eye"></i>
                            </button>
                            <button class="btn-action btn-edit" onclick="changeStatus(${order.id}, '${order.status}')" title="Cambiar estado">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn-action btn-delete" onclick="deleteOrder(${order.id})" title="Eliminar pedido">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
            `).join('');
        } else {
            console.log('⚠️ No hay pedidos o error en datos');
            tbody.innerHTML = `
                <tr>
                    <td colspan="10">
                        <div class="empty-state">
                            <i class="fas fa-shopping-cart"></i>
                            <p>No hay pedidos registrados</p>
                        </div>
                    </td>
                </tr>
            `;
        }
    } catch (error) {
        console.error('❌ Error al cargar pedidos:', error);
        const tbody = document.getElementById('ordersTableBody');
        tbody.innerHTML = `
            <tr>
                <td colspan="10">
                    <div class="empty-state">
                        <i class="fas fa-exclamation-triangle"></i>
                        <p>Error al cargar los pedidos</p>
                        <button class="btn-create" onclick="loadOrders()" style="margin-top: 1rem;">
                            <i class="fas fa-sync"></i>
                            Reintentar
                        </button>
                    </div>
                </td>
            </tr>
        `;
        Swal.fire({
            title: 'Error',
            text: 'No se pudieron cargar los pedidos: ' + error.message,
            icon: 'error'
        });
    }
}

// Obtener texto del estado en español
function getStatusText(status) {
    const statusTexts = {
        'pending': 'Pendiente',
        'confirmed': 'Confirmado',
        'processing': 'En Preparación',
        'shipped': 'Enviado',
        'delivered': 'Entregado',
        'cancelled': 'Cancelado'
    };
    return statusTexts[status] || status;
}

// Obtener texto del método de pago
function getPaymentMethodText(method) {
    const methodTexts = {
        'efectivo': 'Efectivo',
        'tarjeta': 'Tarjeta',
        'transferencia': 'Transferencia',
        'yape': 'Yape',
        'plin': 'Plin'
    };
    return methodTexts[method] || method;
}

// Ver pedido completo
async function viewOrder(id) {
    try {
        const response = await fetch(`{{ route("admin.orders.show", ":id") }}`.replace(':id', id));
        const result = await response.json();
        
        if (result.success) {
            const order = result.data;
            
            let itemsHtml = order.items.map(item => `
                <tr>
                    <td>${item.product_name}</td>
                    <td>S/ ${parseFloat(item.product_price).toFixed(2)}</td>
                    <td>${item.quantity}</td>
                    <td>S/ ${parseFloat(item.subtotal).toFixed(2)}</td>
                </tr>
            `).join('');
            
            Swal.fire({
                title: `Pedido ${order.order_number}`,
                html: `
                    <div style="text-align: left; max-width: 600px;">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1.5rem;">
                            <div>
                                <h4 style="margin-bottom: 0.5rem; color: #667eea;">Información del Cliente</h4>
                                <p><strong>Nombre:</strong> ${order.full_name}</p>
                                ${order.user ? `<p><strong>Email:</strong> ${order.user.email}</p>` : ''}
                                <p><strong>Teléfono:</strong> ${order.phone}</p>
                                <p><strong>Dirección:</strong> ${order.address}</p>
                                <p><strong>Distrito:</strong> ${order.district}</p>
                            </div>
                            <div>
                                <h4 style="margin-bottom: 0.5rem; color: #667eea;">Información del Pedido</h4>
                                <p><strong>Fecha:</strong> ${order.created_at}</p>
                                <p><strong>Estado:</strong> <span class="badge ${getStatusClass(order.status)}">${getStatusText(order.status)}</span></p>
                                <p><strong>Método de Pago:</strong> ${getPaymentMethodText(order.payment_method)}</p>
                                <p><strong>Subtotal:</strong> S/ ${parseFloat(order.subtotal).toFixed(2)}</p>
                                <p><strong>Envío:</strong> S/ ${parseFloat(order.shipping_cost).toFixed(2)}</p>
                                <p><strong style="color: #667eea;">Total:</strong> <strong style="color: #667eea;">S/ ${parseFloat(order.total).toFixed(2)}</strong></p>
                            </div>
                        </div>
                        
                        <h4 style="margin-bottom: 1rem; color: #667eea;">Items del Pedido</h4>
                        <table style="width: 100%; border-collapse: collapse; margin-bottom: 1rem;">
                            <thead>
                                <tr style="background: #f7fafc;">
                                    <th style="padding: 0.5rem; border: 1px solid #e2e8f0; text-align: left;">Producto</th>
                                    <th style="padding: 0.5rem; border: 1px solid #e2e8f0; text-align: right;">Precio</th>
                                    <th style="padding: 0.5rem; border: 1px solid #e2e8f0; text-align: center;">Cantidad</th>
                                    <th style="padding: 0.5rem; border: 1px solid #e2e8f0; text-align: right;">Subtotal</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${itemsHtml}
                            </tbody>
                        </table>
                    </div>
                `,
                icon: 'info',
                confirmButtonText: 'Cerrar',
                confirmButtonColor: '#667eea',
                width: '800px'
            });
        }
    } catch (error) {
        console.error('Error al ver pedido:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudieron cargar los detalles del pedido',
            icon: 'error'
        });
    }
}

// Obtener clase CSS para el estado
function getStatusClass(status) {
    const classes = {
        'pending': 'badge-warning',
        'confirmed': 'badge-info',
        'processing': 'badge-primary',
        'shipped': 'badge-secondary',
        'delivered': 'badge-success',
        'cancelled': 'badge-danger'
    };
    return classes[status] || 'badge-light';
}

// Cambiar estado del pedido
function changeStatus(id, currentStatus) {
    currentOrderId = id;
    
    Swal.fire({
        title: 'Cambiar Estado del Pedido',
        html: document.getElementById('statusModal').innerHTML,
        showCancelButton: true,
        confirmButtonText: 'Actualizar Estado',
        cancelButtonText: 'Cancelar',
        confirmButtonColor: '#667eea',
        width: '400px',
        didOpen: () => {
            // Seleccionar el estado actual
            Swal.getPopup().querySelector('#orderStatus').value = currentStatus;
        },
        preConfirm: () => {
            const status = Swal.getPopup().querySelector('#orderStatus').value;
            
            if (!status) {
                Swal.showValidationMessage('Debe seleccionar un estado');
                return false;
            }
            
            return { status: status };
        }
    }).then((result) => {
        if (result.isConfirmed) {
            updateOrderStatus(id, result.value.status);
        }
    });
}

// Actualizar estado del pedido
async function updateOrderStatus(id, status) {
    try {
        const response = await fetch(`{{ route("admin.orders.status", ":id") }}`.replace(':id', id), {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({ status: status })
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Actualizado!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadOrders();
            loadStats();
        } else {
            Swal.fire({
                title: 'Error',
                text: result.message,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al actualizar estado:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo actualizar el estado del pedido',
            icon: 'error'
        });
    }
}

// Eliminar pedido
function deleteOrder(id) {
    Swal.fire({
        title: '¿Estás seguro?',
        text: 'Esta acción eliminará el pedido y todos sus items. No se puede deshacer.',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sí, eliminar',
        cancelButtonText: 'Cancelar'
    }).then((result) => {
        if (result.isConfirmed) {
            performDeleteOrder(id);
        }
    });
}

// Realizar eliminación
async function performDeleteOrder(id) {
    try {
        const response = await fetch(`{{ route("admin.orders.destroy", ":id") }}`.replace(':id', id), {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Eliminado!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadOrders();
            loadStats();
        } else {
            Swal.fire({
                title: 'Error',
                text: result.message,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al eliminar pedido:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo eliminar el pedido',
            icon: 'error'
        });
    }
}
</script>
@endpush
