@extends('administrador.include.template')

@section('title', 'Gestión de Productos')

@section('breadcrumb')
    <span class="breadcrumb-item">Productos</span>
    <span class="breadcrumb-item active">Lista de Productos</span>
@endsection

@push('styles')
<style>
    .products-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 2rem;
    }
    
    .products-title {
        font-size: 1.5rem;
        font-weight: 600;
        color: var(--gray-900);
    }
    
    .header-actions {
        display: flex;
        gap: 1rem;
        align-items: center;
    }
    
    .btn-create {
        background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
        color: white;
        border: none;
        padding: 0.75rem 1.5rem;
        border-radius: 8px;
        font-weight: 600;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 0.5rem;
        transition: all 0.3s ease;
    }
    
    .btn-create:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    }
    
    .filter-section {
        background: white;
        border-radius: 12px;
        padding: 1.5rem;
        margin-bottom: 1.5rem;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }
    
    .filter-row {
        display: grid;
        grid-template-columns: 1fr 1fr 1fr auto;
        gap: 1rem;
        align-items: end;
    }
    
    .filter-group {
        display: flex;
        flex-direction: column;
    }
    
    .filter-label {
        font-weight: 600;
        margin-bottom: 0.5rem;
        color: var(--gray-700);
    }
    
    .filter-input, .filter-select {
        padding: 0.75rem;
        border: 2px solid var(--gray-200);
        border-radius: 8px;
        font-size: 0.875rem;
    }
    
    .btn-filter {
        background: var(--success-color);
        color: white;
        border: none;
        padding: 0.75rem 1.5rem;
        border-radius: 8px;
        cursor: pointer;
        height: fit-content;
    }
    
    .products-table {
        background: white;
        border-radius: 12px;
        overflow: hidden;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    }
    
    .table-responsive {
        overflow-x: auto;
    }
    
    .table {
        width: 100%;
        margin-bottom: 0;
        color: var(--gray-700);
    }
    
    .table thead th {
        background: var(--gray-50);
        border-bottom: 1px solid var(--gray-200);
        padding: 1rem;
        font-weight: 600;
        text-transform: uppercase;
        font-size: 0.75rem;
        letter-spacing: 0.5px;
        color: var(--gray-600);
    }
    
    .table tbody td {
        padding: 1rem;
        border-bottom: 1px solid var(--gray-100);
        vertical-align: middle;
    }
    
    .table tbody tr:hover {
        background: var(--gray-50);
    }
    
    .product-image {
        width: 50px;
        height: 50px;
        object-fit: cover;
        border-radius: 8px;
        border: 2px solid var(--gray-200);
        background: #f8fafc;
    }
    
    .product-image.fallback {
        display: flex;
        align-items: center;
        justify-content: center;
        background: #f1f5f9;
        color: #64748b;
        font-size: 1.2rem;
    }
    
    .product-name {
        font-weight: 600;
        color: var(--gray-900);
    }
    
    .product-brand {
        font-size: 0.875rem;
        color: var(--gray-600);
    }
    
    .product-price {
        font-weight: 700;
        color: var(--success-color);
        font-size: 1.1rem;
    }
    
    .category-badge {
        background: rgba(102, 126, 234, 0.1);
        color: var(--primary-color);
        padding: 0.25rem 0.75rem;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 600;
    }
    
    .action-buttons {
        display: flex;
        gap: 0.5rem;
    }
    
    .btn-action {
        width: 32px;
        height: 32px;
        border: none;
        border-radius: 6px;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 0.875rem;
        transition: all 0.3s ease;
    }
    
    .btn-view {
        background: rgba(66, 153, 225, 0.1);
        color: var(--info-color);
    }
    
    .btn-edit {
        background: rgba(72, 187, 120, 0.1);
        color: var(--success-color);
    }
    
    .btn-delete {
        background: rgba(245, 101, 101, 0.1);
        color: var(--danger-color);
    }
    
    .btn-action:hover {
        transform: scale(1.1);
    }
    
    .loading-spinner {
        display: flex;
        justify-content: center;
        align-items: center;
        height: 200px;
    }
    
    .spinner {
        width: 40px;
        height: 40px;
        border: 4px solid var(--gray-200);
        border-top: 4px solid var(--primary-color);
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
    
    .empty-state {
        text-align: center;
        padding: 3rem;
        color: var(--gray-500);
    }
    
    .empty-state i {
        font-size: 3rem;
        margin-bottom: 1rem;
        color: var(--gray-300);
    }
</style>
@endpush

@section('content')
<div class="products-container">
    <div class="products-header">
        <h1 class="products-title">
            <i class="fas fa-box"></i>
            Gestión de Productos
        </h1>
        <div class="header-actions">
            <button class="btn-create" onclick="openCreateModal()">
                <i class="fas fa-plus"></i>
                Nuevo Producto
            </button>
        </div>
    </div>
    
    <!-- Filtros -->
    <div class="filter-section">
        <div class="filter-row">
            <div class="filter-group">
                <label class="filter-label">Buscar Producto</label>
                <input type="text" id="searchProduct" class="filter-input" placeholder="Nombre o descripción...">
            </div>
            <div class="filter-group">
                <label class="filter-label">Categoría</label>
                <select id="filterCategory" class="filter-select">
                    <option value="">Todas las categorías</option>
                </select>
            </div>
            <div class="filter-group">
                <label class="filter-label">Rango de Precio</label>
                <select id="filterPrice" class="filter-select">
                    <option value="">Todos los precios</option>
                    <option value="0-500">S/ 0 - S/ 500</option>
                    <option value="500-1000">S/ 500 - S/ 1,000</option>
                    <option value="1000-2000">S/ 1,000 - S/ 2,000</option>
                    <option value="2000+">S/ 2,000+</option>
                </select>
            </div>
            <button class="btn-filter" onclick="applyFilters()">
                <i class="fas fa-filter"></i>
                Filtrar
            </button>
        </div>
    </div>
    
    <div class="products-table">
        <div class="table-responsive">
            <table class="table" id="productsTable">
                <thead>
                    <tr>
                        <th>Imagen</th>
                        <th>Producto</th>
                        <th>Categoría</th>
                        <th>Precio</th>
                        <th>Descripción</th>
                        <th>Fecha Creación</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="productsTableBody">
                    <tr>
                        <td colspan="7">
                            <div class="loading-spinner">
                                <div class="spinner"></div>
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Crear/Editar Producto -->
<div id="productModal" style="display: none;">
    <form id="productForm">
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1rem;">
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Nombre del Producto *</label>
                <input type="text" id="productName" name="name" 
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="Ej: Refrigeradora Samsung 420L" required>
            </div>
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Categoría *</label>
                <select id="productCategory" name="category_id" 
                        style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" required>
                    <option value="">Seleccionar categoría...</option>
                </select>
            </div>
        </div>
        
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1rem;">
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Precio (S/) *</label>
                <input type="number" id="productPrice" name="price" step="0.01" min="0"
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="0.00" required>
            </div>
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">URL de Imagen</label>
                <input type="url" id="productImage" name="image"
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="https://ejemplo.com/imagen.jpg">
            </div>
        </div>
        
        <div style="margin-bottom: 1rem;">
            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Descripción</label>
            <textarea id="productDescription" name="description" rows="4"
                      style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px; resize: vertical;"
                      placeholder="Descripción del producto..."></textarea>
        </div>
    </form>
</div>
@endsection

@push('scripts')
<script>
let currentProductId = null;
let isEditMode = false;
let allProducts = [];
let categories = [];

// Función para manejar imágenes rotas
function handleImageError(img) {
    if (img.dataset.errorHandled) return; // Evitar bucle infinito
    img.dataset.errorHandled = 'true';
    
    const fallbackDiv = document.createElement('div');
    fallbackDiv.className = 'product-image fallback';
    fallbackDiv.innerHTML = '<i class="fas fa-image"></i>';
    
    img.parentNode.replaceChild(fallbackDiv, img);
}

// Función para generar HTML de imagen segura
function generateImageHtml(imageUrl, productName, className = 'product-image') {
    if (!imageUrl || imageUrl.trim() === '') {
        return `<div class="${className} fallback"><i class="fas fa-image"></i></div>`;
    }
    
    return `<img src="${imageUrl}" class="${className}" alt="${productName}" onerror="handleImageError(this)">`;
}

// Función para manejar imágenes rotas
function handleImageError(img) {
    if (img.dataset.errorHandled) return; // Evitar bucle infinito
    img.dataset.errorHandled = 'true';
    
    const fallbackDiv = document.createElement('div');
    fallbackDiv.className = 'product-image fallback';
    fallbackDiv.innerHTML = '<i class="fas fa-image"></i>';
    
    img.parentNode.replaceChild(fallbackDiv, img);
}

// Función para generar HTML de imagen segura
function generateImageHtml(imageUrl, productName, className = 'product-image') {
    if (!imageUrl || imageUrl.trim() === '') {
        return `<div class="${className} fallback"><i class="fas fa-image"></i></div>`;
    }
    
    return `<img src="${imageUrl}" class="${className}" alt="${productName}" onerror="handleImageError(this)">`;
}

// Cargar productos y categorías al cargar la página
document.addEventListener('DOMContentLoaded', function() {
    loadCategories();
    loadProducts();
});

// Cargar categorías
async function loadCategories() {
    try {
        const response = await fetch('{{ route("admin.products.categories") }}');
        const result = await response.json();
        
        if (result.success) {
            categories = result.data;
            
            // Llenar select de filtro
            const filterSelect = document.getElementById('filterCategory');
            filterSelect.innerHTML = '<option value="">Todas las categorías</option>';
            
            categories.forEach(category => {
                filterSelect.innerHTML += `<option value="${category.id}">${category.name}</option>`;
            });
        }
    } catch (error) {
        console.error('Error al cargar categorías:', error);
    }
}

// Cargar productos
async function loadProducts() {
    try {
        const response = await fetch('{{ route("admin.products.data") }}');
        const result = await response.json();
        
        const tbody = document.getElementById('productsTableBody');
        
        if (result.success && result.data.length > 0) {
            allProducts = result.data;
            displayProducts(allProducts);
        } else {
            tbody.innerHTML = `
                <tr>
                    <td colspan="7">
                        <div class="empty-state">
                            <i class="fas fa-box-open"></i>
                            <p>No hay productos registrados</p>
                            <button class="btn-create" onclick="openCreateModal()" style="margin-top: 1rem;">
                                <i class="fas fa-plus"></i>
                                Crear Primer Producto
                            </button>
                        </div>
                    </td>
                </tr>
            `;
        }
    } catch (error) {
        console.error('Error al cargar productos:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudieron cargar los productos',
            icon: 'error'
        });
    }
}

// Mostrar productos en la tabla
function displayProducts(products) {
    const tbody = document.getElementById('productsTableBody');
    
    tbody.innerHTML = products.map(product => `
        <tr>
            <td>
                ${generateImageHtml(product.image, product.name)}
            </td>
            <td>
                <div class="product-name">${product.name}</div>
            </td>
            <td>
                <span class="category-badge">${product.category}</span>
            </td>
            <td>
                <span class="product-price">S/ ${product.price}</span>
            </td>
            <td>
                ${product.description ? product.description.substring(0, 50) + '...' : 'Sin descripción'}
            </td>
            <td>${product.created_at}</td>
            <td>
                <div class="action-buttons">
                    <button class="btn-action btn-view" onclick="viewProduct(${product.id})" title="Ver detalles">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn-action btn-edit" onclick="editProduct(${product.id})" title="Editar">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn-action btn-delete" onclick="deleteProduct(${product.id})" title="Eliminar">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Aplicar filtros
function applyFilters() {
    const searchTerm = document.getElementById('searchProduct').value.toLowerCase();
    const categoryFilter = document.getElementById('filterCategory').value;
    const priceFilter = document.getElementById('filterPrice').value;
    
    let filteredProducts = allProducts.filter(product => {
        // Filtro de búsqueda
        const matchesSearch = !searchTerm || 
            product.name.toLowerCase().includes(searchTerm) ||
            (product.description && product.description.toLowerCase().includes(searchTerm));
        
        // Filtro de categoría
        const matchesCategory = !categoryFilter || product.category_id == categoryFilter;
        
        // Filtro de precio
        let matchesPrice = true;
        if (priceFilter) {
            const price = parseFloat(product.price_raw);
            switch (priceFilter) {
                case '0-500':
                    matchesPrice = price >= 0 && price <= 500;
                    break;
                case '500-1000':
                    matchesPrice = price > 500 && price <= 1000;
                    break;
                case '1000-2000':
                    matchesPrice = price > 1000 && price <= 2000;
                    break;
                case '2000+':
                    matchesPrice = price > 2000;
                    break;
            }
        }
        
        return matchesSearch && matchesCategory && matchesPrice;
    });
    
    displayProducts(filteredProducts);
}

// Abrir modal para crear producto
async function openCreateModal() {
    // Cargar categorías para el select
    await loadCategoriesForSelect();
    
    isEditMode = false;
    currentProductId = null;
    
    Swal.fire({
        title: 'Nuevo Producto',
        html: document.getElementById('productModal').innerHTML,
        showCancelButton: true,
        confirmButtonText: 'Crear Producto',
        cancelButtonText: 'Cancelar',
        confirmButtonColor: '#667eea',
        width: '800px',
        didOpen: () => {
            // Cargar categorías en el select del modal
            loadCategoriesForSelect();
        },
        preConfirm: () => {
            return validateProductForm();
        }
    }).then((result) => {
        if (result.isConfirmed) {
            createProduct(result.value);
        }
    });
}

// Cargar categorías para el select del modal
async function loadCategoriesForSelect() {
    try {
        const response = await fetch('{{ route("admin.products.categories") }}');
        const result = await response.json();
        
        if (result.success) {
            const select = document.getElementById('productCategory') || Swal.getPopup().querySelector('#productCategory');
            if (select) {
                select.innerHTML = '<option value="">Seleccionar categoría...</option>';
                result.data.forEach(category => {
                    select.innerHTML += `<option value="${category.id}">${category.name}</option>`;
                });
            }
        }
    } catch (error) {
        console.error('Error al cargar categorías:', error);
    }
}

// Validar formulario de producto
function validateProductForm() {
    const popup = Swal.getPopup();
    const name = popup.querySelector('#productName').value.trim();
    const categoryId = popup.querySelector('#productCategory').value;
    const price = popup.querySelector('#productPrice').value;
    const image = popup.querySelector('#productImage').value.trim();
    const description = popup.querySelector('#productDescription').value.trim();
    
    if (!name) {
        Swal.showValidationMessage('El nombre del producto es obligatorio');
        return false;
    }
    
    if (!categoryId) {
        Swal.showValidationMessage('La categoría es obligatoria');
        return false;
    }
    
    if (!price || parseFloat(price) < 0) {
        Swal.showValidationMessage('El precio debe ser un número válido mayor o igual a 0');
        return false;
    }
    
    return {
        name,
        category_id: categoryId,
        price: parseFloat(price),
        image: image || null,
        description: description || null
    };
}

// Crear producto
async function createProduct(data) {
    try {
        const response = await fetch('{{ route("admin.products.store") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Éxito!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadProducts();
        } else {
            let errorMessage = result.message;
            if (result.errors) {
                errorMessage = Object.values(result.errors).flat().join('<br>');
            }
            
            Swal.fire({
                title: 'Error de Validación',
                html: errorMessage,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al crear producto:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo crear el producto',
            icon: 'error'
        });
    }
}

// Ver detalles del producto
async function viewProduct(id) {
    try {
        const response = await fetch(`{{ route("admin.products.show", ":id") }}`.replace(':id', id));
        const result = await response.json();
        
        if (result.success) {
            const product = result.data;
            
            Swal.fire({
                title: product.name,
                html: `
                    <div style="text-align: left; max-height: 400px; overflow-y: auto;">
                        ${product.image ? 
                            `<div style="text-align: center; margin-bottom: 1rem;">
                                <img src="${product.image}" style="width: 100%; max-width: 300px; border-radius: 8px;" alt="${product.name}" onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
                                <div style="display: none; padding: 2rem; background: #f1f5f9; border-radius: 8px; color: #64748b;">
                                    <i class="fas fa-image" style="font-size: 3rem;"></i><br>
                                    <span>Imagen no disponible</span>
                                </div>
                            </div>` : 
                            `<div style="text-align: center; padding: 2rem; background: #f1f5f9; border-radius: 8px; margin-bottom: 1rem; color: #64748b;">
                                <i class="fas fa-image" style="font-size: 3rem;"></i><br>
                                <span>Sin imagen</span>
                            </div>`
                        }
                        <p><strong>Categoría:</strong> ${product.category ? product.category.name : 'Sin categoría'}</p>
                        <p><strong>Precio:</strong> S/ ${product.price}</p>
                        ${product.description ? `<p><strong>Descripción:</strong><br>${product.description}</p>` : ''}
                        <p><strong>Fecha de creación:</strong> ${product.created_at}</p>
                        <p><strong>Última actualización:</strong> ${product.updated_at}</p>
                    </div>
                `,
                icon: 'info',
                confirmButtonText: 'Cerrar',
                confirmButtonColor: '#667eea',
                width: '600px'
            });
        }
    } catch (error) {
        console.error('Error al ver producto:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudieron cargar los detalles',
            icon: 'error'
        });
    }
}

// Editar producto
async function editProduct(id) {
    try {
        const response = await fetch(`{{ route("admin.products.show", ":id") }}`.replace(':id', id));
        const result = await response.json();
        
        if (result.success) {
            const product = result.data;
            isEditMode = true;
            currentProductId = id;
            
            Swal.fire({
                title: 'Editar Producto',
                html: document.getElementById('productModal').innerHTML,
                showCancelButton: true,
                confirmButtonText: 'Actualizar Producto',
                cancelButtonText: 'Cancelar',
                confirmButtonColor: '#667eea',
                width: '800px',
                didOpen: async () => {
                    // Cargar categorías
                    await loadCategoriesForSelect();
                    
                    // Llenar los campos con los datos actuales
                    const popup = Swal.getPopup();
                    popup.querySelector('#productName').value = product.name;
                    popup.querySelector('#productCategory').value = product.category_id;
                    popup.querySelector('#productPrice').value = product.price;
                    popup.querySelector('#productImage').value = product.image || '';
                    popup.querySelector('#productDescription').value = product.description || '';
                },
                preConfirm: () => {
                    return validateProductForm();
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    updateProduct(id, result.value);
                }
            });
        }
    } catch (error) {
        console.error('Error al cargar producto:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo cargar el producto',
            icon: 'error'
        });
    }
}

// Actualizar producto
async function updateProduct(id, data) {
    try {
        const response = await fetch(`{{ route("admin.products.update", ":id") }}`.replace(':id', id), {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Éxito!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadProducts();
        } else {
            let errorMessage = result.message;
            if (result.errors) {
                errorMessage = Object.values(result.errors).flat().join('<br>');
            }
            
            Swal.fire({
                title: 'Error de Validación',
                html: errorMessage,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al actualizar producto:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo actualizar el producto',
            icon: 'error'
        });
    }
}

// Eliminar producto
function deleteProduct(id) {
    const product = allProducts.find(p => p.id === id);
    
    Swal.fire({
        title: '¿Estás seguro?',
        text: `Se eliminará el producto "${product.name}"`,
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sí, eliminar',
        cancelButtonText: 'Cancelar'
    }).then((result) => {
        if (result.isConfirmed) {
            performDeleteProduct(id);
        }
    });
}

// Realizar eliminación
async function performDeleteProduct(id) {
    try {
        const response = await fetch(`{{ route("admin.products.destroy", ":id") }}`.replace(':id', id), {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Eliminado!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadProducts();
        } else {
            Swal.fire({
                title: 'Error',
                text: result.message,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al eliminar producto:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo eliminar el producto',
            icon: 'error'
        });
    }
}

// Búsqueda en tiempo real
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('searchProduct');
    if (searchInput) {
        searchInput.addEventListener('input', applyFilters);
    }
});
</script>
@endpush
