@extends('administrador.include.template')

@section('title', 'Gestión de Administradores')

@section('breadcrumb')
    <span class="breadcrumb-item">Usuarios</span>
    <span class="breadcrumb-item active">Administradores</span>
@endsection

@push('styles')
<style>
    .admins-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 2rem;
    }
    
    .admins-title {
        font-size: 1.5rem;
        font-weight: 600;
        color: var(--gray-900);
    }
    
    .btn-create {
        background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
        color: white;
        border: none;
        padding: 0.75rem 1.5rem;
        border-radius: 8px;
        font-weight: 600;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 0.5rem;
        transition: all 0.3s ease;
    }
    
    .btn-create:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    }
    
    .admins-table {
        background: white;
        border-radius: 12px;
        overflow: hidden;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    }
    
    .table-responsive {
        overflow-x: auto;
    }
    
    .table {
        width: 100%;
        margin-bottom: 0;
        color: var(--gray-700);
    }
    
    .table thead th {
        background: var(--gray-50);
        border-bottom: 1px solid var(--gray-200);
        padding: 1rem;
        font-weight: 600;
        text-transform: uppercase;
        font-size: 0.75rem;
        letter-spacing: 0.5px;
        color: var(--gray-600);
    }
    
    .table tbody td {
        padding: 1rem;
        border-bottom: 1px solid var(--gray-100);
        vertical-align: middle;
    }
    
    .table tbody tr:hover {
        background: var(--gray-50);
    }
    
    .admin-name {
        font-weight: 600;
        color: var(--gray-900);
    }
    
    .admin-email {
        font-size: 0.875rem;
        color: var(--gray-600);
    }
    
    .badge {
        padding: 0.25rem 0.75rem;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 600;
    }
    
    .badge-admin {
        background: rgba(139, 69, 19, 0.1);
        color: #8b4513;
    }
    
    .badge-orders {
        background: rgba(34, 197, 94, 0.1);
        color: var(--success-color);
    }
    
    .action-buttons {
        display: flex;
        gap: 0.5rem;
    }
    
    .btn-action {
        width: 32px;
        height: 32px;
        border: none;
        border-radius: 6px;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 0.875rem;
        transition: all 0.3s ease;
    }
    
    .btn-view {
        background: rgba(66, 153, 225, 0.1);
        color: var(--info-color);
    }
    
    .btn-edit {
        background: rgba(72, 187, 120, 0.1);
        color: var(--success-color);
    }
    
    .btn-password {
        background: rgba(251, 191, 36, 0.1);
        color: #f59e0b;
    }
    
    .btn-delete {
        background: rgba(245, 101, 101, 0.1);
        color: var(--danger-color);
    }
    
    .btn-action:hover {
        transform: scale(1.1);
    }
    
    .loading-spinner {
        display: flex;
        justify-content: center;
        align-items: center;
        height: 200px;
    }
    
    .spinner {
        width: 40px;
        height: 40px;
        border: 4px solid var(--gray-200);
        border-top: 4px solid var(--primary-color);
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
    
    .empty-state {
        text-align: center;
        padding: 3rem;
        color: var(--gray-500);
    }
    
    .empty-state i {
        font-size: 3rem;
        margin-bottom: 1rem;
        color: var(--gray-300);
    }
</style>
@endpush

@section('content')
<div class="admins-container">
    <div class="admins-header">
        <h1 class="admins-title">
            <i class="fas fa-user-shield"></i>
            Gestión de Administradores
        </h1>
        <button class="btn-create" onclick="openCreateModal()">
            <i class="fas fa-user-plus"></i>
            Nuevo Administrador
        </button>
    </div>
    
    <div class="admins-table">
        <div class="table-responsive">
            <table class="table" id="adminsTable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Nombre</th>
                        <th>Email</th>
                        <th>Teléfono</th>
                        <th>Pedidos</th>
                        <th>Fecha Registro</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="adminsTableBody">
                    <tr>
                        <td colspan="7">
                            <div class="loading-spinner">
                                <div class="spinner"></div>
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Crear/Editar Administrador -->
<div id="adminModal" style="display: none;">
    <form id="adminForm">
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1rem;">
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Nombre Completo *</label>
                <input type="text" id="adminName" name="name" 
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="Ej: Juan Pérez González" required>
            </div>
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Email *</label>
                <input type="email" id="adminEmail" name="email" 
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="admin@electromart.com" required>
            </div>
        </div>
        
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1rem;">
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Contraseña *</label>
                <input type="password" id="adminPassword" name="password"
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="Mínimo 6 caracteres" required>
            </div>
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Teléfono</label>
                <input type="tel" id="adminPhone" name="phone"
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="Ej: +51 999 999 999">
            </div>
        </div>
    </form>
</div>

<!-- Modal Cambiar Contraseña -->
<div id="passwordModal" style="display: none;">
    <form id="passwordForm">
        <div style="margin-bottom: 1rem;">
            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Nueva Contraseña *</label>
            <input type="password" id="newPassword" name="password"
                   style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                   placeholder="Mínimo 6 caracteres" required>
        </div>
        <div style="margin-bottom: 1rem;">
            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Confirmar Contraseña *</label>
            <input type="password" id="confirmPassword" name="password_confirmation"
                   style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                   placeholder="Repetir la contraseña" required>
        </div>
    </form>
</div>
@endsection

@push('scripts')
<script>
let currentAdminId = null;
let isEditMode = false;

// Cargar administradores al cargar la página
document.addEventListener('DOMContentLoaded', function() {
    loadAdmins();
});

// Cargar administradores
async function loadAdmins() {
    console.log('🔄 Cargando administradores...');
    try {
        const response = await fetch('{{ route("admin.users.admins.data") }}');
        const result = await response.json();
        
        const tbody = document.getElementById('adminsTableBody');
        
        if (result.success && result.data.length > 0) {
            console.log('✅ Administradores encontrados:', result.data.length);
            tbody.innerHTML = result.data.map(admin => `
                <tr>
                    <td>${admin.id}</td>
                    <td>
                        <div class="admin-name">${admin.name}</div>
                    </td>
                    <td>
                        <div class="admin-email">${admin.email}</div>
                    </td>
                    <td>${admin.phone}</td>
                    <td>
                        <span class="badge badge-orders">${admin.orders_count} pedidos</span>
                    </td>
                    <td>${admin.created_at}</td>
                    <td>
                        <div class="action-buttons">
                            <button class="btn-action btn-view" onclick="viewAdmin(${admin.id})" title="Ver detalles">
                                <i class="fas fa-eye"></i>
                            </button>
                            <button class="btn-action btn-edit" onclick="editAdmin(${admin.id})" title="Editar">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn-action btn-password" onclick="changePassword(${admin.id})" title="Cambiar contraseña">
                                <i class="fas fa-key"></i>
                            </button>
                            <button class="btn-action btn-delete" onclick="deleteAdmin(${admin.id})" title="Eliminar">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
            `).join('');
        } else {
            console.log('⚠️ No hay administradores registrados');
            tbody.innerHTML = `
                <tr>
                    <td colspan="7">
                        <div class="empty-state">
                            <i class="fas fa-user-shield"></i>
                            <p>No hay administradores registrados</p>
                            <button class="btn-create" onclick="openCreateModal()" style="margin-top: 1rem;">
                                <i class="fas fa-user-plus"></i>
                                Crear Primer Administrador
                            </button>
                        </div>
                    </td>
                </tr>
            `;
        }
    } catch (error) {
        console.error('❌ Error al cargar administradores:', error);
        const tbody = document.getElementById('adminsTableBody');
        tbody.innerHTML = `
            <tr>
                <td colspan="7">
                    <div class="empty-state">
                        <i class="fas fa-exclamation-triangle"></i>
                        <p>Error al cargar los administradores</p>
                        <button class="btn-create" onclick="loadAdmins()" style="margin-top: 1rem;">
                            <i class="fas fa-sync"></i>
                            Reintentar
                        </button>
                    </div>
                </td>
            </tr>
        `;
        Swal.fire({
            title: 'Error',
            text: 'No se pudieron cargar los administradores: ' + error.message,
            icon: 'error'
        });
    }
}

// Abrir modal para crear administrador
function openCreateModal() {
    isEditMode = false;
    currentAdminId = null;
    
    Swal.fire({
        title: 'Nuevo Administrador',
        html: document.getElementById('adminModal').innerHTML,
        showCancelButton: true,
        confirmButtonText: 'Crear Administrador',
        cancelButtonText: 'Cancelar',
        confirmButtonColor: '#667eea',
        width: '600px',
        preConfirm: () => {
            return validateAdminForm();
        }
    }).then((result) => {
        if (result.isConfirmed) {
            createAdmin(result.value);
        }
    });
}

// Validar formulario de administrador
function validateAdminForm() {
    const popup = Swal.getPopup();
    const name = popup.querySelector('#adminName').value.trim();
    const email = popup.querySelector('#adminEmail').value.trim();
    const password = popup.querySelector('#adminPassword').value;
    const phone = popup.querySelector('#adminPhone').value.trim();
    
    if (!name) {
        Swal.showValidationMessage('El nombre es obligatorio');
        return false;
    }
    
    if (!email) {
        Swal.showValidationMessage('El email es obligatorio');
        return false;
    }
    
    if (!isEditMode && !password) {
        Swal.showValidationMessage('La contraseña es obligatoria');
        return false;
    }
    
    if (password && password.length < 6) {
        Swal.showValidationMessage('La contraseña debe tener al menos 6 caracteres');
        return false;
    }
    
    const data = {
        name,
        email,
        phone: phone || null,
        rol: 'administrador'
    };
    
    if (password) {
        data.password = password;
    }
    
    return data;
}

// Crear administrador
async function createAdmin(data) {
    try {
        const response = await fetch('{{ route("admin.users.store") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Éxito!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadAdmins();
        } else {
            let errorMessage = result.message;
            if (result.errors) {
                errorMessage = Object.values(result.errors).flat().join('<br>');
            }
            
            Swal.fire({
                title: 'Error de Validación',
                html: errorMessage,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al crear administrador:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo crear el administrador',
            icon: 'error'
        });
    }
}

// Ver detalles del administrador
async function viewAdmin(id) {
    try {
        const response = await fetch(`{{ route("admin.users.show", ":id") }}`.replace(':id', id));
        const result = await response.json();
        
        if (result.success) {
            const admin = result.data;
            
            Swal.fire({
                title: admin.name,
                html: `
                    <div style="text-align: left;">
                        <p><strong>ID:</strong> ${admin.id}</p>
                        <p><strong>Email:</strong> ${admin.email}</p>
                        <p><strong>Teléfono:</strong> ${admin.phone || 'No especificado'}</p>
                        <p><strong>Rol:</strong> <span class="badge badge-admin">Administrador</span></p>
                        <p><strong>Pedidos gestionados:</strong> ${admin.orders_count}</p>
                        <p><strong>Fecha de registro:</strong> ${admin.created_at}</p>
                        <p><strong>Última actualización:</strong> ${admin.updated_at}</p>
                    </div>
                `,
                icon: 'info',
                confirmButtonText: 'Cerrar',
                confirmButtonColor: '#667eea',
                width: '500px'
            });
        }
    } catch (error) {
        console.error('Error al ver administrador:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudieron cargar los detalles',
            icon: 'error'
        });
    }
}

// Editar administrador
async function editAdmin(id) {
    try {
        const response = await fetch(`{{ route("admin.users.show", ":id") }}`.replace(':id', id));
        const result = await response.json();
        
        if (result.success) {
            const admin = result.data;
            isEditMode = true;
            currentAdminId = id;
            
            Swal.fire({
                title: 'Editar Administrador',
                html: document.getElementById('adminModal').innerHTML,
                showCancelButton: true,
                confirmButtonText: 'Actualizar Administrador',
                cancelButtonText: 'Cancelar',
                confirmButtonColor: '#667eea',
                width: '600px',
                didOpen: () => {
                    const popup = Swal.getPopup();
                    popup.querySelector('#adminName').value = admin.name;
                    popup.querySelector('#adminEmail').value = admin.email;
                    popup.querySelector('#adminPhone').value = admin.phone || '';
                    popup.querySelector('#adminPassword').placeholder = 'Dejar vacío para no cambiar';
                    popup.querySelector('#adminPassword').required = false;
                },
                preConfirm: () => {
                    return validateAdminForm();
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    updateAdmin(id, result.value);
                }
            });
        }
    } catch (error) {
        console.error('Error al cargar administrador:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo cargar el administrador',
            icon: 'error'
        });
    }
}

// Actualizar administrador
async function updateAdmin(id, data) {
    try {
        const response = await fetch(`{{ route("admin.users.update", ":id") }}`.replace(':id', id), {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Éxito!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadAdmins();
        } else {
            let errorMessage = result.message;
            if (result.errors) {
                errorMessage = Object.values(result.errors).flat().join('<br>');
            }
            
            Swal.fire({
                title: 'Error de Validación',
                html: errorMessage,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al actualizar administrador:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo actualizar el administrador',
            icon: 'error'
        });
    }
}

// Cambiar contraseña
function changePassword(id) {
    Swal.fire({
        title: 'Cambiar Contraseña',
        html: document.getElementById('passwordModal').innerHTML,
        showCancelButton: true,
        confirmButtonText: 'Cambiar Contraseña',
        cancelButtonText: 'Cancelar',
        confirmButtonColor: '#f59e0b',
        width: '500px',
        preConfirm: () => {
            const popup = Swal.getPopup();
            const password = popup.querySelector('#newPassword').value;
            const confirmation = popup.querySelector('#confirmPassword').value;
            
            if (!password) {
                Swal.showValidationMessage('La nueva contraseña es obligatoria');
                return false;
            }
            
            if (password.length < 6) {
                Swal.showValidationMessage('La contraseña debe tener al menos 6 caracteres');
                return false;
            }
            
            if (password !== confirmation) {
                Swal.showValidationMessage('Las contraseñas no coinciden');
                return false;
            }
            
            return { password, password_confirmation: confirmation };
        }
    }).then((result) => {
        if (result.isConfirmed) {
            performChangePassword(id, result.value);
        }
    });
}

// Realizar cambio de contraseña
async function performChangePassword(id, data) {
    try {
        const response = await fetch(`{{ route("admin.users.password", ":id") }}`.replace(':id', id), {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Éxito!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
        } else {
            let errorMessage = result.message;
            if (result.errors) {
                errorMessage = Object.values(result.errors).flat().join('<br>');
            }
            
            Swal.fire({
                title: 'Error',
                html: errorMessage,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al cambiar contraseña:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo cambiar la contraseña',
            icon: 'error'
        });
    }
}

// Eliminar administrador
function deleteAdmin(id) {
    Swal.fire({
        title: '¿Estás seguro?',
        text: 'Esta acción no se puede deshacer',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sí, eliminar',
        cancelButtonText: 'Cancelar'
    }).then((result) => {
        if (result.isConfirmed) {
            performDeleteAdmin(id);
        }
    });
}

// Realizar eliminación
async function performDeleteAdmin(id) {
    try {
        const response = await fetch(`{{ route("admin.users.destroy", ":id") }}`.replace(':id', id), {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Eliminado!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadAdmins();
        } else {
            Swal.fire({
                title: 'Error',
                text: result.message,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al eliminar administrador:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo eliminar el administrador',
            icon: 'error'
        });
    }
}
</script>
@endpush
