@extends('administrador.include.template')

@section('title', 'Gestión de Clientes')

@section('breadcrumb')
    <span class="breadcrumb-item">Usuarios</span>
    <span class="breadcrumb-item active">Clientes</span>
@endsection

@push('styles')
<style>
    .clients-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 2rem;
    }
    
    .clients-title {
        font-size: 1.5rem;
        font-weight: 600;
        color: var(--gray-900);
    }
    
    .header-actions {
        display: flex;
        gap: 1rem;
        align-items: center;
    }
    
    .btn-create {
        background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
        color: white;
        border: none;
        padding: 0.75rem 1.5rem;
        border-radius: 8px;
        font-weight: 600;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 0.5rem;
        transition: all 0.3s ease;
    }
    
    .btn-create:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    }
    
    .filter-section {
        background: white;
        border-radius: 12px;
        padding: 1.5rem;
        margin-bottom: 1.5rem;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }
    
    .filter-row {
        display: grid;
        grid-template-columns: 1fr 1fr auto;
        gap: 1rem;
        align-items: end;
    }
    
    .filter-group {
        display: flex;
        flex-direction: column;
    }
    
    .filter-label {
        font-weight: 600;
        margin-bottom: 0.5rem;
        color: var(--gray-700);
    }
    
    .filter-input {
        padding: 0.75rem;
        border: 2px solid var(--gray-200);
        border-radius: 8px;
        font-size: 0.875rem;
    }
    
    .btn-filter {
        background: var(--success-color);
        color: white;
        border: none;
        padding: 0.75rem 1.5rem;
        border-radius: 8px;
        cursor: pointer;
        height: fit-content;
    }
    
    .clients-table {
        background: white;
        border-radius: 12px;
        overflow: hidden;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    }
    
    .table-responsive {
        overflow-x: auto;
    }
    
    .table {
        width: 100%;
        margin-bottom: 0;
        color: var(--gray-700);
    }
    
    .table thead th {
        background: var(--gray-50);
        border-bottom: 1px solid var(--gray-200);
        padding: 1rem;
        font-weight: 600;
        text-transform: uppercase;
        font-size: 0.75rem;
        letter-spacing: 0.5px;
        color: var(--gray-600);
    }
    
    .table tbody td {
        padding: 1rem;
        border-bottom: 1px solid var(--gray-100);
        vertical-align: middle;
    }
    
    .table tbody tr:hover {
        background: var(--gray-50);
    }
    
    .client-name {
        font-weight: 600;
        color: var(--gray-900);
    }
    
    .client-email {
        font-size: 0.875rem;
        color: var(--gray-600);
    }
    
    .badge {
        padding: 0.25rem 0.75rem;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 600;
    }
    
    .badge-client {
        background: rgba(59, 130, 246, 0.1);
        color: var(--info-color);
    }
    
    .badge-orders {
        background: rgba(34, 197, 94, 0.1);
        color: var(--success-color);
    }
    
    .action-buttons {
        display: flex;
        gap: 0.5rem;
    }
    
    .btn-action {
        width: 32px;
        height: 32px;
        border: none;
        border-radius: 6px;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 0.875rem;
        transition: all 0.3s ease;
    }
    
    .btn-view {
        background: rgba(66, 153, 225, 0.1);
        color: var(--info-color);
    }
    
    .btn-edit {
        background: rgba(72, 187, 120, 0.1);
        color: var(--success-color);
    }
    
    .btn-password {
        background: rgba(251, 191, 36, 0.1);
        color: #f59e0b;
    }
    
    .btn-delete {
        background: rgba(245, 101, 101, 0.1);
        color: var(--danger-color);
    }
    
    .btn-action:hover {
        transform: scale(1.1);
    }
    
    .loading-spinner {
        display: flex;
        justify-content: center;
        align-items: center;
        height: 200px;
    }
    
    .spinner {
        width: 40px;
        height: 40px;
        border: 4px solid var(--gray-200);
        border-top: 4px solid var(--primary-color);
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
    
    .empty-state {
        text-align: center;
        padding: 3rem;
        color: var(--gray-500);
    }
    
    .empty-state i {
        font-size: 3rem;
        margin-bottom: 1rem;
        color: var(--gray-300);
    }
</style>
@endpush

@section('content')
<div class="clients-container">
    <div class="clients-header">
        <h1 class="clients-title">
            <i class="fas fa-users"></i>
            Gestión de Clientes
        </h1>
        <div class="header-actions">
            <button class="btn-create" onclick="openCreateModal()">
                <i class="fas fa-user-plus"></i>
                Nuevo Cliente
            </button>
        </div>
    </div>
    
    <!-- Filtros -->
    <div class="filter-section">
        <div class="filter-row">
            <div class="filter-group">
                <label class="filter-label">Buscar Cliente</label>
                <input type="text" id="searchClient" class="filter-input" placeholder="Nombre o email...">
            </div>
            <div class="filter-group">
                <label class="filter-label">Pedidos</label>
                <select id="filterOrders" class="filter-input">
                    <option value="">Todos</option>
                    <option value="0">Sin pedidos</option>
                    <option value="1-5">1-5 pedidos</option>
                    <option value="6+">6+ pedidos</option>
                </select>
            </div>
            <button class="btn-filter" onclick="applyFilters()">
                <i class="fas fa-filter"></i>
                Filtrar
            </button>
        </div>
    </div>
    
    <div class="clients-table">
        <div class="table-responsive">
            <table class="table" id="clientsTable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Cliente</th>
                        <th>Email</th>
                        <th>Teléfono</th>
                        <th>Pedidos</th>
                        <th>Fecha Registro</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="clientsTableBody">
                    <tr>
                        <td colspan="7">
                            <div class="loading-spinner">
                                <div class="spinner"></div>
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Crear/Editar Cliente -->
<div id="clientModal" style="display: none;">
    <form id="clientForm">
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1rem;">
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Nombre Completo *</label>
                <input type="text" id="clientName" name="name" 
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="Ej: María García López" required>
            </div>
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Email *</label>
                <input type="email" id="clientEmail" name="email" 
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="cliente@gmail.com" required>
            </div>
        </div>
        
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1rem;">
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Contraseña *</label>
                <input type="password" id="clientPassword" name="password"
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="Mínimo 6 caracteres" required>
            </div>
            <div>
                <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Teléfono</label>
                <input type="tel" id="clientPhone" name="phone"
                       style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                       placeholder="Ej: +51 999 999 999">
            </div>
        </div>
    </form>
</div>

<!-- Modal Cambiar Contraseña -->
<div id="passwordModal" style="display: none;">
    <form id="passwordForm">
        <div style="margin-bottom: 1rem;">
            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Nueva Contraseña *</label>
            <input type="password" id="newPassword" name="password"
                   style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                   placeholder="Mínimo 6 caracteres" required>
        </div>
        <div style="margin-bottom: 1rem;">
            <label style="display: block; margin-bottom: 0.5rem; font-weight: 600;">Confirmar Contraseña *</label>
            <input type="password" id="confirmPassword" name="password_confirmation"
                   style="width: 100%; padding: 0.75rem; border: 2px solid #e2e8f0; border-radius: 8px;" 
                   placeholder="Repetir la contraseña" required>
        </div>
    </form>
</div>
@endsection

@push('scripts')
<script>
let currentClientId = null;
let isEditMode = false;
let allClients = [];

// Cargar clientes al cargar la página
document.addEventListener('DOMContentLoaded', function() {
    loadClients();
});

// Cargar clientes
async function loadClients() {
    console.log('🔄 Cargando clientes...');
    try {
        const response = await fetch('{{ route("admin.users.clients.data") }}');
        const result = await response.json();
        
        const tbody = document.getElementById('clientsTableBody');
        
        if (result.success && result.data.length > 0) {
            console.log('✅ Clientes encontrados:', result.data.length);
            allClients = result.data;
            displayClients(allClients);
        } else {
            console.log('⚠️ No hay clientes registrados');
            tbody.innerHTML = `
                <tr>
                    <td colspan="7">
                        <div class="empty-state">
                            <i class="fas fa-users"></i>
                            <p>No hay clientes registrados</p>
                            <button class="btn-create" onclick="openCreateModal()" style="margin-top: 1rem;">
                                <i class="fas fa-user-plus"></i>
                                Crear Primer Cliente
                            </button>
                        </div>
                    </td>
                </tr>
            `;
        }
    } catch (error) {
        console.error('❌ Error al cargar clientes:', error);
        const tbody = document.getElementById('clientsTableBody');
        tbody.innerHTML = `
            <tr>
                <td colspan="7">
                    <div class="empty-state">
                        <i class="fas fa-exclamation-triangle"></i>
                        <p>Error al cargar los clientes</p>
                        <button class="btn-create" onclick="loadClients()" style="margin-top: 1rem;">
                            <i class="fas fa-sync"></i>
                            Reintentar
                        </button>
                    </div>
                </td>
            </tr>
        `;
        Swal.fire({
            title: 'Error',
            text: 'No se pudieron cargar los clientes: ' + error.message,
            icon: 'error'
        });
    }
}

// Mostrar clientes en la tabla
function displayClients(clients) {
    const tbody = document.getElementById('clientsTableBody');
    
    tbody.innerHTML = clients.map(client => `
        <tr>
            <td>${client.id}</td>
            <td>
                <div class="client-name">${client.name}</div>
            </td>
            <td>
                <div class="client-email">${client.email}</div>
            </td>
            <td>${client.phone}</td>
            <td>
                <span class="badge badge-orders">${client.orders_count} pedidos</span>
            </td>
            <td>${client.created_at}</td>
            <td>
                <div class="action-buttons">
                    <button class="btn-action btn-view" onclick="viewClient(${client.id})" title="Ver detalles">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn-action btn-edit" onclick="editClient(${client.id})" title="Editar">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn-action btn-password" onclick="changePassword(${client.id})" title="Cambiar contraseña">
                        <i class="fas fa-key"></i>
                    </button>
                    <button class="btn-action btn-delete" onclick="deleteClient(${client.id})" title="Eliminar">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Aplicar filtros
function applyFilters() {
    const searchTerm = document.getElementById('searchClient').value.toLowerCase();
    const ordersFilter = document.getElementById('filterOrders').value;
    
    let filteredClients = allClients.filter(client => {
        // Filtro de búsqueda
        const matchesSearch = !searchTerm || 
            client.name.toLowerCase().includes(searchTerm) ||
            client.email.toLowerCase().includes(searchTerm);
        
        // Filtro de pedidos
        let matchesOrders = true;
        if (ordersFilter) {
            const ordersCount = client.orders_count;
            switch (ordersFilter) {
                case '0':
                    matchesOrders = ordersCount === 0;
                    break;
                case '1-5':
                    matchesOrders = ordersCount >= 1 && ordersCount <= 5;
                    break;
                case '6+':
                    matchesOrders = ordersCount >= 6;
                    break;
            }
        }
        
        return matchesSearch && matchesOrders;
    });
    
    displayClients(filteredClients);
}

// Abrir modal para crear cliente
function openCreateModal() {
    isEditMode = false;
    currentClientId = null;
    
    Swal.fire({
        title: 'Nuevo Cliente',
        html: document.getElementById('clientModal').innerHTML,
        showCancelButton: true,
        confirmButtonText: 'Crear Cliente',
        cancelButtonText: 'Cancelar',
        confirmButtonColor: '#667eea',
        width: '600px',
        preConfirm: () => {
            return validateClientForm();
        }
    }).then((result) => {
        if (result.isConfirmed) {
            createClient(result.value);
        }
    });
}

// Validar formulario de cliente
function validateClientForm() {
    const popup = Swal.getPopup();
    const name = popup.querySelector('#clientName').value.trim();
    const email = popup.querySelector('#clientEmail').value.trim();
    const password = popup.querySelector('#clientPassword').value;
    const phone = popup.querySelector('#clientPhone').value.trim();
    
    if (!name) {
        Swal.showValidationMessage('El nombre es obligatorio');
        return false;
    }
    
    if (!email) {
        Swal.showValidationMessage('El email es obligatorio');
        return false;
    }
    
    if (!isEditMode && !password) {
        Swal.showValidationMessage('La contraseña es obligatoria');
        return false;
    }
    
    if (password && password.length < 6) {
        Swal.showValidationMessage('La contraseña debe tener al menos 6 caracteres');
        return false;
    }
    
    const data = {
        name,
        email,
        phone: phone || null,
        rol: 'cliente'
    };
    
    if (password) {
        data.password = password;
    }
    
    return data;
}

// Crear cliente
async function createClient(data) {
    try {
        const response = await fetch('{{ route("admin.users.store") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Éxito!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadClients();
        } else {
            let errorMessage = result.message;
            if (result.errors) {
                errorMessage = Object.values(result.errors).flat().join('<br>');
            }
            
            Swal.fire({
                title: 'Error de Validación',
                html: errorMessage,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al crear cliente:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo crear el cliente',
            icon: 'error'
        });
    }
}

// Ver detalles del cliente
async function viewClient(id) {
    try {
        const response = await fetch(`{{ route("admin.users.show", ":id") }}`.replace(':id', id));
        const result = await response.json();
        
        if (result.success) {
            const client = result.data;
            
            Swal.fire({
                title: client.name,
                html: `
                    <div style="text-align: left;">
                        <p><strong>ID:</strong> ${client.id}</p>
                        <p><strong>Email:</strong> ${client.email}</p>
                        <p><strong>Teléfono:</strong> ${client.phone || 'No especificado'}</p>
                        <p><strong>Rol:</strong> <span class="badge badge-client">Cliente</span></p>
                        <p><strong>Pedidos realizados:</strong> ${client.orders_count}</p>
                        <p><strong>Fecha de registro:</strong> ${client.created_at}</p>
                        <p><strong>Última actualización:</strong> ${client.updated_at}</p>
                    </div>
                `,
                icon: 'info',
                confirmButtonText: 'Cerrar',
                confirmButtonColor: '#667eea',
                width: '500px'
            });
        }
    } catch (error) {
        console.error('Error al ver cliente:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudieron cargar los detalles',
            icon: 'error'
        });
    }
}

// Editar cliente
async function editClient(id) {
    try {
        const response = await fetch(`{{ route("admin.users.show", ":id") }}`.replace(':id', id));
        const result = await response.json();
        
        if (result.success) {
            const client = result.data;
            isEditMode = true;
            currentClientId = id;
            
            Swal.fire({
                title: 'Editar Cliente',
                html: document.getElementById('clientModal').innerHTML,
                showCancelButton: true,
                confirmButtonText: 'Actualizar Cliente',
                cancelButtonText: 'Cancelar',
                confirmButtonColor: '#667eea',
                width: '600px',
                didOpen: () => {
                    const popup = Swal.getPopup();
                    popup.querySelector('#clientName').value = client.name;
                    popup.querySelector('#clientEmail').value = client.email;
                    popup.querySelector('#clientPhone').value = client.phone || '';
                    popup.querySelector('#clientPassword').placeholder = 'Dejar vacío para no cambiar';
                    popup.querySelector('#clientPassword').required = false;
                },
                preConfirm: () => {
                    return validateClientForm();
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    updateClient(id, result.value);
                }
            });
        }
    } catch (error) {
        console.error('Error al cargar cliente:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo cargar el cliente',
            icon: 'error'
        });
    }
}

// Actualizar cliente
async function updateClient(id, data) {
    try {
        const response = await fetch(`{{ route("admin.users.update", ":id") }}`.replace(':id', id), {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Éxito!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadClients();
        } else {
            let errorMessage = result.message;
            if (result.errors) {
                errorMessage = Object.values(result.errors).flat().join('<br>');
            }
            
            Swal.fire({
                title: 'Error de Validación',
                html: errorMessage,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al actualizar cliente:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo actualizar el cliente',
            icon: 'error'
        });
    }
}

// Cambiar contraseña
function changePassword(id) {
    Swal.fire({
        title: 'Cambiar Contraseña',
        html: document.getElementById('passwordModal').innerHTML,
        showCancelButton: true,
        confirmButtonText: 'Cambiar Contraseña',
        cancelButtonText: 'Cancelar',
        confirmButtonColor: '#f59e0b',
        width: '500px',
        preConfirm: () => {
            const popup = Swal.getPopup();
            const password = popup.querySelector('#newPassword').value;
            const confirmation = popup.querySelector('#confirmPassword').value;
            
            if (!password) {
                Swal.showValidationMessage('La nueva contraseña es obligatoria');
                return false;
            }
            
            if (password.length < 6) {
                Swal.showValidationMessage('La contraseña debe tener al menos 6 caracteres');
                return false;
            }
            
            if (password !== confirmation) {
                Swal.showValidationMessage('Las contraseñas no coinciden');
                return false;
            }
            
            return { password, password_confirmation: confirmation };
        }
    }).then((result) => {
        if (result.isConfirmed) {
            performChangePassword(id, result.value);
        }
    });
}

// Realizar cambio de contraseña
async function performChangePassword(id, data) {
    try {
        const response = await fetch(`{{ route("admin.users.password", ":id") }}`.replace(':id', id), {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Éxito!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
        } else {
            let errorMessage = result.message;
            if (result.errors) {
                errorMessage = Object.values(result.errors).flat().join('<br>');
            }
            
            Swal.fire({
                title: 'Error',
                html: errorMessage,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al cambiar contraseña:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo cambiar la contraseña',
            icon: 'error'
        });
    }
}

// Eliminar cliente
function deleteClient(id) {
    const client = allClients.find(c => c.id === id);
    
    Swal.fire({
        title: '¿Estás seguro?',
        text: `Se eliminará el cliente "${client.name}"`,
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sí, eliminar',
        cancelButtonText: 'Cancelar'
    }).then((result) => {
        if (result.isConfirmed) {
            performDeleteClient(id);
        }
    });
}

// Realizar eliminación
async function performDeleteClient(id) {
    try {
        const response = await fetch(`{{ route("admin.users.destroy", ":id") }}`.replace(':id', id), {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        });
        
        const result = await response.json();
        
        if (result.success) {
            Swal.fire({
                title: '¡Eliminado!',
                text: result.message,
                icon: 'success',
                timer: 2000,
                showConfirmButton: false
            });
            loadClients();
        } else {
            Swal.fire({
                title: 'Error',
                text: result.message,
                icon: 'error'
            });
        }
    } catch (error) {
        console.error('Error al eliminar cliente:', error);
        Swal.fire({
            title: 'Error',
            text: 'No se pudo eliminar el cliente',
            icon: 'error'
        });
    }
}

// Búsqueda en tiempo real
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('searchClient');
    if (searchInput) {
        searchInput.addEventListener('input', applyFilters);
    }
});
</script>
@endpush
