<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\PrincipalController;
use App\Http\Controllers\ProductoController;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\UserController;
use App\Http\Controllers\ContactController;
use App\Http\Controllers\AdminController;
use App\Http\Controllers\BlogController;
/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

Route::get('/', [PrincipalController::class, 'index'])->name('index');
Route::get('/home', [PrincipalController::class, 'index'])->name('home');
Route::resource('Producto', ProductoController::class);
Route::get('/products/filter', [ProductoController::class, 'filterProducts'])->name('products.filter');
Route::get('/categories', [ProductoController::class, 'getCategories'])->name('categories.get');

// Rutas de carritoprocessOrder
Route::post('/add-to-cart', [ProductoController::class, 'addToCart'])->name('cart.add');
Route::get('/cart-count', [ProductoController::class, 'getCartCount'])->name('cart.count');
Route::get('/cart-items', [ProductoController::class, 'getCartItems'])->name('cart.items');
Route::post('/update-cart-item', [ProductoController::class, 'updateCartItem'])->name('cart.update');
Route::post('/remove-cart-item', [ProductoController::class, 'removeCartItem'])->name('cart.remove');
Route::get('/checkout', [ProductoController::class, 'checkout'])->name('checkout');
Route::post('/process-order', [ProductoController::class, 'processOrder'])->name('order.process');
Route::get('/order-success/{order}', [ProductoController::class, 'orderSuccess'])->name('order.success');
Route::post('/clear-order-session', [ProductoController::class, 'clearOrderSession'])->name('order.clear.session');

// Ruta temporal para debug - limpiar sesión
Route::get('/debug/clear-session', function() {
    Session::forget(['processing_order', 'order_completed', 'last_order_id']);
    return redirect()->route('index')->with('success', 'Sesión limpiada');
})->name('debug.clear.session');

// Rutas de autenticación
Route::get('/login', [AuthController::class, 'showLogin'])->name('login');
Route::post('/login', [AuthController::class, 'login'])->name('login.post');
Route::get('/register', [AuthController::class, 'showRegister'])->name('register');
Route::post('/register', [AuthController::class, 'register'])->name('register.post');
Route::post('/logout', [AuthController::class, 'logout'])->name('logout');
Route::get('/check-auth', [AuthController::class, 'checkAuth'])->name('check.auth');
Route::post('/create-test-user', [AuthController::class, 'createTestUser'])->name('create.test.user');

// Rutas de perfil de usuario
Route::get('/mi-perfil', [UserController::class, 'profile'])->name('user.profile');
Route::get('/mis-pedidos', [UserController::class, 'orders'])->name('user.orders');
Route::get('/configuracion', [UserController::class, 'settings'])->name('user.settings');
Route::post('/perfil/actualizar', [UserController::class, 'updateProfile'])->name('user.profile.update');
Route::post('/perfil/cambiar-password', [UserController::class, 'updatePassword'])->name('user.password.update');

// Rutas de contacto y cotizaciones
Route::post('/contact/send-message', [ContactController::class, 'sendMessage'])->name('contact.send.message');
Route::post('/contact/request-quote', [ContactController::class, 'requestQuote'])->name('contact.request.quote');
Route::get('/contact/company-info', [ContactController::class, 'getCompanyInfo'])->name('contact.company.info');

// Admin routes (proteger con middleware de admin más adelante)
Route::get('/admin/messages', [ContactController::class, 'indexMessages'])->name('admin.messages');
Route::get('/admin/quotes', [ContactController::class, 'indexQuotes'])->name('admin.quotes');
Route::post('/admin/message/{id}/read', [ContactController::class, 'markMessageAsRead'])->name('admin.message.read');
Route::post('/admin/quote/{id}/status', [ContactController::class, 'updateQuoteStatus'])->name('admin.quote.status');

// Rutas de administración de empresa (compatibilidad)
Route::get('/admin/company-info', [AdminController::class, 'companyInfo'])->name('admin.company.info');
Route::post('/admin/company-info', [AdminController::class, 'updateCompanyInfo'])->name('admin.company.info.update');
Route::get('/admin/messages-full', [AdminController::class, 'messages'])->name('admin.messages.full');
Route::get('/admin/quotes-full', [AdminController::class, 'quotes'])->name('admin.quotes.full');

// Rutas del blog
Route::get('/blog', [BlogController::class, 'index'])->name('blog.index');
Route::get('/blog/{slug}', [BlogController::class, 'show'])->name('blog.show');
Route::get('/api/blog/featured', [BlogController::class, 'getFeaturedPosts'])->name('blog.api.featured');

// ===== RUTAS DEL PANEL DE ADMINISTRACIÓN =====
// Rutas públicas del admin (login)
Route::get('/administrador', [AdminController::class, 'showLogin'])->name('admin.login');
Route::post('/administrador/login', [AdminController::class, 'login'])->name('admin.login.post');

// Rutas protegidas del admin (requieren autenticación de administrador)
Route::prefix('administrador')->middleware('auth')->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\Admin\DashboardController::class, 'index'])->name('admin.index');
    Route::post('/logout', [AdminController::class, 'logout'])->name('admin.logout');
    
    // Rutas del dashboard con datos
    Route::prefix('dashboard')->name('admin.dashboard.')->group(function () {
        Route::get('/stats', [App\Http\Controllers\Admin\DashboardController::class, 'getStats'])->name('stats');
        Route::get('/sales-chart', [App\Http\Controllers\Admin\DashboardController::class, 'getSalesChart'])->name('sales-chart');
        Route::get('/orders-status-chart', [App\Http\Controllers\Admin\DashboardController::class, 'getOrdersStatusChart'])->name('orders-status-chart');
        Route::get('/top-products', [App\Http\Controllers\Admin\DashboardController::class, 'getTopProducts'])->name('top-products');
        Route::get('/category-sales', [App\Http\Controllers\Admin\DashboardController::class, 'getCategorySales'])->name('category-sales');
        Route::get('/recent-orders', [App\Http\Controllers\Admin\DashboardController::class, 'getRecentOrders'])->name('recent-orders');
    });
    
    // Rutas heredadas del AdminController (mantener compatibilidad)
    Route::get('/stats', [AdminController::class, 'getStats'])->name('admin.stats');
    Route::get('/sales-data', [AdminController::class, 'getSalesData'])->name('admin.sales.data');
    
    // Rutas de gestión de categorías
    Route::prefix('categories')->name('admin.categories.')->group(function () {
        Route::get('/', [App\Http\Controllers\Admin\CategoryController::class, 'index'])->name('index');
        Route::get('/data', [App\Http\Controllers\Admin\CategoryController::class, 'getData'])->name('data');
        Route::post('/', [App\Http\Controllers\Admin\CategoryController::class, 'store'])->name('store');
        Route::get('/{id}', [App\Http\Controllers\Admin\CategoryController::class, 'show'])->name('show');
        Route::put('/{id}', [App\Http\Controllers\Admin\CategoryController::class, 'update'])->name('update');
        Route::delete('/{id}', [App\Http\Controllers\Admin\CategoryController::class, 'destroy'])->name('destroy');
        Route::get('/select/list', [App\Http\Controllers\Admin\CategoryController::class, 'getForSelect'])->name('select');
    });
    
    // Rutas de gestión de productos
    Route::prefix('products')->name('admin.products.')->group(function () {
        Route::get('/', [App\Http\Controllers\Admin\ProductController::class, 'index'])->name('index');
        Route::get('/data', [App\Http\Controllers\Admin\ProductController::class, 'getData'])->name('data');
        Route::post('/', [App\Http\Controllers\Admin\ProductController::class, 'store'])->name('store');
        Route::get('/categories', [App\Http\Controllers\Admin\ProductController::class, 'getCategories'])->name('categories');
        Route::get('/{id}', [App\Http\Controllers\Admin\ProductController::class, 'show'])->name('show');
        Route::put('/{id}', [App\Http\Controllers\Admin\ProductController::class, 'update'])->name('update');
        Route::delete('/{id}', [App\Http\Controllers\Admin\ProductController::class, 'destroy'])->name('destroy');
    });
    
    // Rutas de gestión de usuarios
    Route::prefix('users')->name('admin.users.')->group(function () {
        // Vistas de administradores y clientes
        Route::get('/admins', [App\Http\Controllers\Admin\UserController::class, 'indexAdmins'])->name('admins');
        Route::get('/clients', [App\Http\Controllers\Admin\UserController::class, 'indexClients'])->name('clients');
        
        // API endpoints para obtener datos
        Route::get('/admins/data', [App\Http\Controllers\Admin\UserController::class, 'getAdminsData'])->name('admins.data');
        Route::get('/clients/data', [App\Http\Controllers\Admin\UserController::class, 'getClientsData'])->name('clients.data');
        
        // CRUD operations
        Route::post('/', [App\Http\Controllers\Admin\UserController::class, 'store'])->name('store');
        Route::get('/{id}', [App\Http\Controllers\Admin\UserController::class, 'show'])->name('show');
        Route::put('/{id}', [App\Http\Controllers\Admin\UserController::class, 'update'])->name('update');
        Route::delete('/{id}', [App\Http\Controllers\Admin\UserController::class, 'destroy'])->name('destroy');
        
        // Cambio de contraseña
        Route::put('/{id}/password', [App\Http\Controllers\Admin\UserController::class, 'changePassword'])->name('password');
    });
    
    // Rutas de gestión del blog
    Route::prefix('blog')->name('admin.blog.')->group(function () {
        Route::get('/', [BlogController::class, 'adminIndex'])->name('index');
        Route::get('/data', [BlogController::class, 'getData'])->name('data');
        Route::post('/', [BlogController::class, 'store'])->name('store');
        Route::get('/categories', [BlogController::class, 'getCategories'])->name('categories');
        Route::get('/{id}', [BlogController::class, 'showForEdit'])->name('show');
        Route::put('/{id}', [BlogController::class, 'update'])->name('update');
        Route::delete('/{id}', [BlogController::class, 'destroy'])->name('destroy');
        Route::post('/{id}/toggle-featured', [BlogController::class, 'toggleFeatured'])->name('toggle-featured');
    });
    
    // Rutas de gestión de mensajes de contacto
    Route::prefix('messages')->name('admin.messages.')->group(function () {
        Route::get('/', [App\Http\Controllers\Admin\MessageController::class, 'index'])->name('index');
        Route::get('/data', [App\Http\Controllers\Admin\MessageController::class, 'getData'])->name('data');
        Route::get('/{id}', [App\Http\Controllers\Admin\MessageController::class, 'show'])->name('show');
        Route::delete('/{id}', [App\Http\Controllers\Admin\MessageController::class, 'destroy'])->name('destroy');
        Route::put('/{id}/answered', [App\Http\Controllers\Admin\MessageController::class, 'markAsAnswered'])->name('answered');
    });
    
    // Rutas de gestión de pedidos
    Route::prefix('orders')->name('admin.orders.')->group(function () {
        Route::get('/', [App\Http\Controllers\Admin\OrderController::class, 'index'])->name('index');
        Route::get('/data', [App\Http\Controllers\Admin\OrderController::class, 'getData'])->name('data');
        Route::get('/stats', [App\Http\Controllers\Admin\OrderController::class, 'getStats'])->name('stats');
        Route::get('/{id}', [App\Http\Controllers\Admin\OrderController::class, 'show'])->name('show');
        Route::put('/{id}/status', [App\Http\Controllers\Admin\OrderController::class, 'updateStatus'])->name('status');
        Route::delete('/{id}', [App\Http\Controllers\Admin\OrderController::class, 'destroy'])->name('destroy');
    });
    
    // Rutas de gestión de cotizaciones
    Route::prefix('quotes')->name('admin.quotes.')->group(function () {
        Route::get('/', [App\Http\Controllers\Admin\QuoteController::class, 'index'])->name('index');
        Route::get('/data', [App\Http\Controllers\Admin\QuoteController::class, 'getData'])->name('data');
        Route::get('/stats', [App\Http\Controllers\Admin\QuoteController::class, 'getStats'])->name('stats');
        Route::get('/{id}', [App\Http\Controllers\Admin\QuoteController::class, 'show'])->name('show');
        Route::put('/{id}/status', [App\Http\Controllers\Admin\QuoteController::class, 'updateStatus'])->name('update-status');
        Route::put('/{id}/priority', [App\Http\Controllers\Admin\QuoteController::class, 'updatePriority'])->name('update-priority');
        Route::delete('/{id}', [App\Http\Controllers\Admin\QuoteController::class, 'destroy'])->name('destroy');
    });
    
    // Rutas de gestión de información de la empresa
    Route::prefix('company-info')->name('admin.company-info.')->group(function () {
        Route::get('/', [App\Http\Controllers\Admin\CompanyInfoController::class, 'index'])->name('index');
        Route::get('/data', [App\Http\Controllers\Admin\CompanyInfoController::class, 'getData'])->name('data');
        Route::post('/update', [App\Http\Controllers\Admin\CompanyInfoController::class, 'update'])->name('update');
    });
    
    // Rutas de reportes
    Route::prefix('reports')->name('admin.reports.')->group(function () {
        Route::get('/', [App\Http\Controllers\Admin\ReportController::class, 'index'])->name('index');
        Route::get('/orders', [App\Http\Controllers\Admin\ReportController::class, 'ordersReport'])->name('orders');
        Route::get('/messages', [App\Http\Controllers\Admin\ReportController::class, 'messagesReport'])->name('messages');
        Route::get('/products', [App\Http\Controllers\Admin\ReportController::class, 'productsReport'])->name('products');
        Route::get('/users', [App\Http\Controllers\Admin\ReportController::class, 'usersReport'])->name('users');
    });
});

// Ruta para crear administrador (solo desarrollo)
Route::get('/create-admin', function() {
    return view('create-admin');
})->name('admin.create.form');
Route::post('/create-admin', [AdminController::class, 'createAdmin'])->name('admin.create');

// Admin routes originales (compatibilidad)
Route::get('/admin/messages', [ContactController::class, 'indexMessages'])->name('admin.messages');
Route::get('/admin/quotes', [ContactController::class, 'indexQuotes'])->name('admin.quotes');
Route::post('/admin/message/{id}/read', [ContactController::class, 'markMessageAsRead'])->name('admin.message.read');
Route::post('/admin/quote/{id}/status', [ContactController::class, 'updateQuoteStatus'])->name('admin.quote.status');

// Rutas de administración de empresa (mantener solo estas)
Route::get('/admin/company-info', [AdminController::class, 'companyInfo'])->name('admin.company.info');
Route::post('/admin/company-info', [AdminController::class, 'updateCompanyInfo'])->name('admin.company.info.update');
Route::get('/admin/messages-full', [AdminController::class, 'messages'])->name('admin.messages.full');
Route::get('/admin/quotes-full', [AdminController::class, 'quotes'])->name('admin.quotes.full');
